/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.workflow.appointment.repeat;

import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;

import java.util.Objects;

/**
 * Calendar block series.
 *
 * @author Tim Anderson
 */
public class CalendarBlockSeries extends ScheduleEventSeries {

    /**
     * Constructs an {@link CalendarBlockSeries}.
     *
     * @param event   the event. A calendar block
     * @param service the archetype service
     */
    public CalendarBlockSeries(Act event, IArchetypeService service) {
        super(event, service);
    }

    /**
     * Determines if the series can be calculated.
     *
     * @param state the current event state
     * @return {@code true} if the series can be calculated
     */
    @Override
    protected boolean canCalculateSeries(State state) {
        return super.canCalculateSeries(state) && ((BlockState) state).getBlockType() != null;
    }

    /**
     * Creates state from an act.
     *
     * @param bean the act bean
     * @return a new state
     */
    @Override
    protected State createState(IMObjectBean bean) {
        return new BlockState(bean);
    }

    /**
     * Copies state.
     *
     * @param state the state to copy
     * @return a copy of {@code state}
     */
    @Override
    protected State copy(State state) {
        return new BlockState((BlockState) state);
    }

    /**
     * Populates an event from state. This is invoked after the event times and schedule have been set.
     *
     * @param bean  the event bean
     * @param state the state
     */
    @Override
    protected void populate(IMObjectBean bean, State state) {
        super.populate(bean, state);
        BlockState block = (BlockState) state;
        bean.setTarget("type", block.getBlockType());
        bean.setValue("name", block.getName());
        bean.setValue("notes", block.getNotes());
    }

    private static class BlockState extends State {

        /**
         * The block type reference.
         */
        private Reference blockType;

        /**
         * The name.
         */
        private String name;

        /**
         * The notes.
         */
        private String notes;

        /**
         * Initialises the state from an event.
         *
         * @param event the event
         */
        public BlockState(IMObjectBean event) {
            super(event);
        }

        /**
         * Copy constructor.
         *
         * @param state the state to copy
         */
        public BlockState(BlockState state) {
            super(state);
            blockType = state.blockType;
            name = state.getName();
            notes = state.getNotes();
        }

        /**
         * Updates the state from an event.
         *
         * @param event the event
         */
        @Override
        public void update(IMObjectBean event) {
            super.update(event);
            blockType = event.getTargetRef("type");
            name = event.getString("name");
            notes = event.getString("notes");
        }

        /**
         * Returns the block type.
         *
         * @return the block type
         */
        public Reference getBlockType() {
            return blockType;
        }

        /**
         * Returns the block name.
         *
         * @return the block name. May be {@code null}
         */
        public String getName() {
            return name;
        }

        /**
         * Returns the block notes.
         *
         * @return the block notes. May be {@code null}
         */
        public String getNotes() {
            return notes;
        }

        /**
         * Indicates whether some other object is "equal to" this one.
         *
         * @param obj the reference object with which to compare.
         * @return {@code true} if this object is the same as the obj
         */
        @Override
        public boolean equals(Object obj) {
            boolean result = false;
            if (obj instanceof BlockState && super.equals(obj)) {
                BlockState other = (BlockState) obj;
                result = Objects.equals(blockType, other.blockType)
                         && Objects.equals(name, other.name)
                         && Objects.equals(notes, other.notes);
            }
            return result;
        }

    }
}
