/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.payment.paymentprocessor;

import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.domain.service.object.DomainObjectService;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.ViewActActions;
import org.openvpms.web.component.im.query.QueryBrowser;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.payment.PaymentProcessorStatusDialog;
import org.openvpms.web.workspace.reporting.payment.TransactionCRUDWindow;

/**
 * CRUD window to view payment processor transaction.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorCRUDWindow extends TransactionCRUDWindow {

    /**
     * Payment processor send link button identifier.
     */
    private static final String SEND_LINK = "button.sendLink";

    /**
     * Constructs a {@link PaymentProcessorCRUDWindow}.
     *
     * @param archetypes the transaction archetypes
     * @param browser    the browser
     * @param context    the context
     * @param help       the help context
     */
    public PaymentProcessorCRUDWindow(Archetypes<FinancialAct> archetypes, QueryBrowser<FinancialAct> browser,
                                      Context context, HelpContext help) {
        super(archetypes, new TransactionActions(), browser, context, help);
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        super.layoutButtons(buttons);
        buttons.add(createPrintButton());
        buttons.add(SEND_LINK, action(this::sendLink, (object) -> getActions().canSendLink(object)));
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        super.enableButtons(buttons, enable);
        buttons.setEnabled(PRINT_ID, enable);
        buttons.setEnabled(SEND_LINK, enable && getActions().canSendLink(getObject()));
    }

    /**
     * Determines the actions that may be performed on the selected object.
     *
     * @return the actions
     */
    @Override
    protected TransactionActions getActions() {
        return (TransactionActions) super.getActions();
    }

    /**
     * Sends a payment processor link.
     *
     * @param object the payment processor transaction
     */
    private void sendLink(FinancialAct object) {
        DomainObjectService service = ServiceHelper.getBean(DomainObjectService.class);
        Transaction transaction = service.create(object, Transaction.class);
        PaymentProcessorStatusDialog dialog = PaymentProcessorStatusDialog.create(
                transaction.getPaymentProcessor(), transaction, object,
                createLayoutContext(getHelpContext()), true);
        dialog.show();
    }


    protected static class TransactionActions extends ViewActActions<FinancialAct> {

        /**
         * Determines if a payment processor link can be sent.
         *
         * @param act the transaction
         * @return {@code true} if a link can be sent, otherwise {@code false}
         */
        public boolean canSendLink(FinancialAct act) {
            boolean result = false;
            if (act != null && act.isA(PaymentProcessorArchetypes.PAYMENT, PaymentProcessorArchetypes.REFUND)
                && PaymentProcessorTransactionStatus.SUBMITTED.equals(act.getStatus())) {
                IMObjectBean bean = getBean(act);
                result = bean.getString("url") != null;
            }
            return result;
        }
    }
}
