/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.patient.summary;

import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.action.ActionFactory;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.query.Browser;
import org.openvpms.web.component.im.query.BrowserDialog;
import org.openvpms.web.component.im.query.BrowserFactory;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.query.QueryFactory;
import org.openvpms.web.echo.dialog.PopupDialogListener;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

import java.util.function.Consumer;

/**
 * Edits the cage on a patient.
 *
 * @author Tim Anderson
 */
public class PatientCageEditor {

    /**
     * Remove cage button id.
     */
    private static final String REMOVE_CAGE_ID = "button.removeCage";

    /**
     * Edits the cage on a patient
     *
     * @param patient            the patient
     * @param context            the context
     * @param help               the help context
     * @param completionListener the listener to be notified when the patient is updated
     */
    public void edit(Party patient, Context context, HelpContext help, Consumer<Party> completionListener) {
        BrowserDialog<Lookup> dialog = createCageBrowserDialog(context, help);
        dialog.addWindowPaneListener(new PopupDialogListener() {
            @Override
            public void onOK() {
                Lookup lookup = dialog.getSelected();
                update(patient, lookup, completionListener);
            }

            /**
             * Invoked when an unhandled button is selected.
             *
             * @param action the dialog action
             */
            @Override
            public void onAction(String action) {
                if (REMOVE_CAGE_ID.equals(action)) {
                    update(patient, null, completionListener);
                }
            }
        });
        dialog.show();
    }

    /**
     * Creates a browser dialog of cages.
     *
     * @param context the context
     * @param help    the help context
     * @return a new dialog
     */
    private BrowserDialog<Lookup> createCageBrowserDialog(Context context, HelpContext help) {
        Query<Lookup> query = QueryFactory.create(PatientArchetypes.CAGE, context);
        Browser<Lookup> browser = BrowserFactory.create(query, new DefaultLayoutContext(context, help));
        String displayName = DescriptorHelper.getDisplayName(PatientArchetypes.CAGE,
                                                             ServiceHelper.getArchetypeService());
        String title = Messages.format("imobject.select.title", displayName);
        String[] buttons = {BrowserDialog.CANCEL_ID, REMOVE_CAGE_ID};
        BrowserDialog<Lookup> dialog = new BrowserDialog<>(title, buttons, browser, help);
        dialog.resize("BrowserDialog.size.lookup.cage");
        return dialog;
    }

    /**
     * Updates the cage on a patient.
     *
     * @param patient            the patient
     * @param lookup             the cage lookup. May be {@code null}
     * @param completionListener the listener to notify on successful update
     */
    private void update(Party patient, Lookup lookup, Consumer<Party> completionListener) {
        ServiceHelper.getBean(ActionFactory.class).newAction()
                .withLatest(patient).asBean(bean -> {
                    bean.setValue("cage", lookup != null ? lookup.getCode() : null);
                    bean.save();
                    completionListener.accept(bean.getObject(Party.class));
                }).run();
    }
}