/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.act.ActStatusHelper;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.object.SequencedRelationship;
import org.openvpms.web.component.im.delete.Deletable;
import org.openvpms.web.component.im.edit.payment.PaymentItemEditor;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.property.DefaultValidator;
import org.openvpms.web.component.property.MutableProperty;
import org.openvpms.web.component.property.ValidationHelper;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.resource.i18n.Messages;

import java.util.List;


/**
 * An editor for <em>act.customerAccountPayment*</em> and  <em>act.customerAccountRefund*</em>
 * where the payment/refund is managed via 3rd party transactions.
 *
 * @author Tim Anderson
 */
public abstract class TransactionPaymentItemEditor extends PaymentItemEditor {

    /**
     * Determines if there is a prior transaction for the item.
     */
    private boolean hasTransaction;

    /**
     * Determines if the payment/refund has is complete. If so, no further transactions are required.
     */
    private boolean complete;

    /**
     * Transactions node.
     */
    private static final String TRANSACTIONS = "transactions";

    /**
     * Constructs {@link TransactionPaymentItemEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent act
     * @param context the layout context
     */
    public TransactionPaymentItemEditor(FinancialAct act, FinancialAct parent, LayoutContext context) {
        super(act, parent, context);
    }

    /**
     * Determines if the payment item requires a transaction to be completed before the parent can be POSTED.
     *
     * @return {@code true} if the payment requires a transaction to be completed, otherwise {@code false}
     */
    public boolean requiresTransaction() {
        return !isComplete();
    }

    /**
     * Performs a transaction.
     *
     * @param listener the listener to notify on success
     */
    public void performTransaction(Runnable listener) {
        Validator validator = new DefaultValidator();
        if (validate(validator)) {
            TransactionState state = getTransactionState();
            if (state != null) {
                performTransaction(state, listener);
            } else {
                // transaction is now complete
                refreshLayout();
                listener.run();
            }
        } else {
            ValidationHelper.showError(validator);
        }
    }

    /**
     * Determines if the editor can be deleted.
     * <p/>
     * The editor can be deleted if there is no transaction in progress, and none has been approved.
     *
     * @return {@code true} if the editor can be deleted
     */
    @Override
    public boolean canDelete() {
        return getDeletable().canDelete();
    }

    /**
     * Determines if the current transaction can be cancelled.
     *
     * @return {@code true} if current transaction can be cancelled, otherwise {@code false}
     */
    public boolean canCancelTransaction() {
        return false;
    }

    /**
     * Cancels the current transaction.
     */
    public void cancelTransaction() {
        throw new IllegalStateException("Transaction cancellation not supported");
    }

    /**
     * Determines if this is deletable.
     * <p/>
     * This examines all transactions to determine if they allow or prevent deletion.
     *
     * @return {@link Deletable#yes()} if the payment item can be deleted, otherwise {@link Deletable#no}
     */
    public Deletable getDeletable() {
        Deletable result = Deletable.yes();
        for (FinancialAct act : getTransactions()) {
            result = getDeletable(act);
            if (!result.canDelete()) {
                break;
            }
        }
        return result;
    }

    /**
     * Returns the transaction display name, for use in error messages.
     *
     * @return the transaction display name
     */
    protected abstract String getTransactionDisplayName();

    /**
     * Performs a transaction.
     * <p/>
     * This is only invoked if the item is valid.
     *
     * @param state    the transaction state
     * @param listener the listener to notify on success
     */
    protected abstract void performTransaction(TransactionState state, Runnable listener);

    /**
     * Determines if a transaction allows or prevents deletion of this payment item.
     *
     * @return {@link Deletable#yes()} if the payment item can be deleted, otherwise {@link Deletable#no}
     */
    protected abstract Deletable getDeletable(FinancialAct transaction);

    /**
     * Determines if a new transaction is required.
     *
     * @param transaction the last transaction
     * @return {@code true} if a new transaction is required, otherwise {@code false}
     */
    protected abstract boolean isNewTransactionRequired(Act transaction);

    /**
     * Determines if a transaction is complete. If it is complete, no further transactions may be performed.
     *
     * @param transaction the transaction
     * @return {@code true} if the transaction is complete, otherwise {@code false}
     */
    protected abstract boolean isComplete(FinancialAct transaction);

    /**
     * Determines if the transaction is complete.
     *
     * @return {@code true} if the transaction is complete, otherwise {@code false}
     */
    protected boolean isComplete() {
        return complete;
    }

    /**
     * Determines if a transaction is present.
     *
     * @return {@code true} if a transaction is present, otherwise {@code false}
     */
    protected boolean hasTransaction() {
        return hasTransaction;
    }

    /**
     * Refreshes the layout.
     * <p/>
     * This implementation invokes {@link #updateTransactionStatus()} first, to determine if fields should be read-only
     */
    protected void refreshLayout() {
        updateTransactionStatus();
        onLayout();
    }

    /**
     * Validates the object.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        return super.doValidation(validator) && validateStatus(validator);
    }

    /**
     * Returns the transactions, ordered on increasing sequence.
     *
     * @return the transactions
     */
    protected List<FinancialAct> getTransactions() {
        return getBean(getObject()).getTargets(TRANSACTIONS, FinancialAct.class, Policies.orderBySequence());
    }

    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        return new TransactionPaymentLayoutStrategy();
    }

    /**
     * Adds a transaction.
     *
     * @param act      the transaction act
     * @param sequence the transaction sequence
     */
    protected void addTransaction(FinancialAct act, int sequence) {
        IMObjectBean bean = getBean(getObject());
        SequencedRelationship relationship = (SequencedRelationship) bean.addTarget(TRANSACTIONS, act, "transaction");
        relationship.setSequence(sequence);
        bean.save(act);
    }

    /**
     * Updates the transaction status.
     * <p/>
     * This should be invoked on construction.
     * <p/>
     * This retrieves the last transaction, if any, to determine if:
     * <ul>
     *     <li>a transaction is required</li>
     *     <li>fields should be read only</li>
     * </ul>
     */
    protected void updateTransactionStatus() {
        Act parent = (Act) getParent();
        if (ActStatusHelper.isPosted(parent, getService())) {
            // editing a POSTED payment. Can only occur through administration. No changes are possible,
            // so just flag it as complete
            updateTransactionStatus(null, true);
        } else {
            FinancialAct transaction = getLastTransaction();
            if (transaction != null) {
                boolean complete = isComplete(transaction);
                updateTransactionStatus(transaction, complete);
            } else {
                updateTransactionStatus(null, false);
            }
        }
    }

    /**
     * Updates the transaction status.
     *
     * @param transaction the last transaction. May be {@code null}
     * @param complete    if {@code true}, payment/refund is complete, and no further transactions will be performed
     * @return {@code true} if the layout needs to be updated, otherwise {@code false}
     */
    protected boolean updateTransactionStatus(FinancialAct transaction, boolean complete) {
        boolean changed = this.complete != complete;
        this.complete = complete;
        hasTransaction = transaction != null;
        changed |= updateReadOnly(AMOUNT, markPropertiesReadOnly());
        return changed;
    }

    /**
     * Validates the status.
     * <p/>
     * This will be considered invalid if the parent act is POSTED, but an EFT transaction is required.
     *
     * @param validator the validator
     * @return {@code true} if the status is valid, otherwise {@code false}
     */
    protected boolean validateStatus(Validator validator) {
        boolean result;
        Act parent = (Act) getParent();
        if (ActStatus.POSTED.equals(parent.getStatus()) && requiresTransaction()) {
            validator.add(this, Messages.format("customer.payment.postedWithOutstandingTransaction",
                                                getDisplayName(parent), getTransactionDisplayName()));
            result = false;
        } else {
            result = true;
        }
        return result;
    }

    /**
     * Returns the last transaction.
     *
     * @return the last transaction, or {@code null} if none exists
     */
    protected FinancialAct getLastTransaction() {
        TransactionState lastTransaction = getLastTransaction(getBean(getObject()));
        return lastTransaction != null ? lastTransaction.getAct() : null;
    }

    /**
     * Updates the read-only status of a property.
     * <p/>
     * The property must be a {@link MutableProperty}.
     *
     * @param name     the property name
     * @param readOnly if {@code true}, the property is read-only, else it is editable
     * @return {@code true} if the property was changed, otherwise {@code false}
     */
    protected boolean updateReadOnly(String name, boolean readOnly) {
        boolean changed = false;
        MutableProperty property = (MutableProperty) getProperty(name);
        if (property.isReadOnly() != readOnly && !property.isHidden()) {
            changed = property.setReadOnly(readOnly);
        }
        return changed;
    }

    /**
     * Determines if properties should be marked read-only.
     *
     * @return {@code true} if properties should be marked read-only, otherwise {@code false}
     */
    protected boolean markPropertiesReadOnly() {
        return isComplete();
    }

    /**
     * Returns the current transaction state.
     * <p/>
     * This returns:
     * <ul>
     *  <li>The most recent transaction, if it is incomplete; or</li>
     *  <li>The next sequence, if a new transaction is required; or</li>
     *  <li>No state, if the transaction has been performed successfully.</li>
     * </ul>
     *
     * @return the transaction state, or {@code null} if no further transaction is required
     */
    private TransactionState getTransactionState() {
        TransactionState result = null;
        IMObjectBean bean = getBean(getObject());
        TransactionState last = getLastTransaction(bean);
        int sequence = 0;
        if (last != null) {
            FinancialAct transaction = last.getAct();
            sequence = last.getSequence();
            if (isNewTransactionRequired(transaction)) {
                result = new TransactionState(null, ++sequence);
            } else if (!isComplete(transaction)) {
                result = last;
            }
        } else {
            result = new TransactionState(null, ++sequence);
        }
        return result;
    }

    /**
     * Returns the last transaction and its relationship sequence.
     *
     * @return the transaction state, or {@code null} if none exists
     */
    private TransactionState getLastTransaction(IMObjectBean bean) {
        int sequence = 0;
        Reference last = null;
        for (SequencedRelationship relationship : bean.getValues(TRANSACTIONS, SequencedRelationship.class)) {
            if (relationship.getSequence() >= sequence) {
                last = relationship.getTarget();
                sequence = relationship.getSequence();
            }
        }

        FinancialAct transaction = null;
        if (last != null) {
            transaction = bean.getObject(last, FinancialAct.class);
        }
        return transaction != null ? new TransactionState(transaction, sequence) : null;
    }

    protected static class TransactionState {

        private final FinancialAct act;

        private final int sequence;

        public TransactionState(FinancialAct act, int sequence) {
            this.act = act;
            this.sequence = sequence;
        }

        public FinancialAct getAct() {
            return act;
        }

        public int getSequence() {
            return sequence;
        }
    }
}