/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;

import org.openvpms.archetype.rules.doc.DocumentTemplate;
import org.openvpms.archetype.rules.doc.EmailTemplate;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.paymentprocessor.transaction.Payment;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.LocalContext;
import org.openvpms.web.component.im.contact.ContactHelper;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.report.ContextDocumentTemplateLocator;
import org.openvpms.web.component.im.sms.SMSDialog;
import org.openvpms.web.component.im.sms.SMSTemplateEvaluator;
import org.openvpms.web.component.mail.EmailTemplateEvaluator;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.component.mail.MailDialog;
import org.openvpms.web.component.mail.MailEditor;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.dialog.MessageDialog;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.admin.template.paymentprocessor.PaymentProcessorSMSEvaluator;
import org.openvpms.web.workspace.customer.CustomerMailContext;

import java.util.List;

/**
 * Dialog to display a payment processor transaction status.
 * <p/>
 * If the transaction includes a URL, this can be sent to the customer via mail or SMS.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorStatusDialog extends MessageDialog {

    /**
     * The transaction.
     */
    private final FinancialAct transaction;

    /**
     * The URL.
     */
    private final String url;

    /**
     * The layout context.
     */
    private final LayoutContext context;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The payment processor.
     */
    private final Entity processor;

    /**
     * The mail button identifier.
     */
    private static final String MAIL_ID = "button.mail";

    /**
     * The SMS button identifier.
     */
    private static final String SMS_ID = "button.sms.send";

    /**
     * Constructs a {@link PaymentProcessorStatusDialog}.
     *
     * @param paymentProcessor the payment processor
     * @param transaction      the payment processor transaction
     * @param message          the dialog message
     * @param context          the layout context
     */
    private PaymentProcessorStatusDialog(Entity paymentProcessor, FinancialAct transaction, String message, String url,
                                         ArchetypeService service, LayoutContext context) {
        super(paymentProcessor.getName(), message, OK);
        this.transaction = transaction;
        this.service = service;
        this.url = url;
        this.processor = paymentProcessor;
        this.context = context;
        if (PaymentProcessorTransactionStatus.SUBMITTED.equals(transaction.getStatus())) {
            if (url != null) {
                addButton(MAIL_ID);
                addButton(SMS_ID);
            }
        }
    }

    /**
     * Creates a new dialog.
     *
     * @param paymentProcessor the payment processor
     * @param transaction      the transaction
     * @param act              the transaction act
     * @param context          the layout context
     * @param sendLink         if {@code true}, the add a Send Link title to the dialog, otherwise use the
     *                         payment processor name
     * @return a new dialog
     */
    public static PaymentProcessorStatusDialog create(Entity paymentProcessor, Transaction transaction,
                                                      FinancialAct act, LayoutContext context, boolean sendLink) {
        ArchetypeService service = ServiceHelper.getArchetypeService();
        StringBuilder builder = new StringBuilder();
        Transaction.Status status = transaction.getStatus();
        if (status == Transaction.Status.SUBMITTED) {
            if (transaction instanceof Payment) {
                builder.append(Messages.format("customer.payment.pp.payment.SUBMITTED", paymentProcessor.getName()));
            } else {
                builder.append(Messages.format("customer.payment.pp.refund.SUBMITTED", paymentProcessor.getName()));
            }
            builder.append("\n\n");
            String email = transaction.getEmail();
            if (email != null) {
                // at the moment, all supported payment processors email a link to the customer.
                builder.append(Messages.format("customer.payment.pp.email", paymentProcessor.getName(), email));
            } else {
                builder.append(Messages.format("customer.payment.pp.sendLink", paymentProcessor.getName()));
            }
        } else {
            String key = "customer.payment.pp." + status;
            String statusMessage;
            if (status == Transaction.Status.ERROR) {
                String message = transaction.getMessage();
                if (message == null) {
                    statusMessage = Messages.get(key + ".nomessage");
                } else {
                    statusMessage = Messages.format(key, message);
                }
            } else {
                statusMessage = Messages.format(key, paymentProcessor.getName());
            }
            builder.append(statusMessage);
        }
        String message = builder.toString();
        PaymentProcessorStatusDialog dialog = new PaymentProcessorStatusDialog(paymentProcessor, act, message,
                                                                               transaction.getUrl(), service, context);
        if (sendLink) {
            dialog.setTitle(Messages.get("customer.payment.pp.sendLink.title"));
        }
        return dialog;
    }

    /**
     * Invoked when a button is pressed. This delegates to the appropriate
     * on*() method for the button if it is known, else sets the action to
     * the button identifier and closes the window.
     *
     * @param button the button identifier
     */
    @Override
    protected void onButton(String button) {
        if (button.equals(MAIL_ID)) {
            onMail();
        } else if (button.equals(SMS_ID)) {
            onSMS();
        } else {
            super.onButton(button);
        }
    }

    /**
     * Send a mail containing the transaction URL to the customer.
     */
    private void onMail() {
        MailContext mailContext = context.getMailContext();
        if (!(mailContext instanceof CustomerMailContext)) {
            mailContext = new CustomerMailContext(context.getContext());
        }
        MailDialog dialog = new MailDialog(mailContext, null, context);
        dialog.show();
        // set the message after the dialog is shown, so if the template cannot be evaluated the error will
        // be displayed above the dialog

        MailEditor mailEditor = dialog.getMailEditor();
        EmailTemplate template = getEmailTemplate();
        String subject = null;
        String message = null;
        if (template != null) {
            try {
                Context localContext = LocalContext.copy(context.getContext());
                IMObjectBean bean = service.getBean(transaction);
                localContext.setLocation(bean.getTarget("location", Party.class));
                localContext.setCustomer(bean.getTarget("customer", Party.class));

                EmailTemplateEvaluator evaluator = ServiceHelper.getBean(EmailTemplateEvaluator.class);
                subject = evaluator.getSubject(template, transaction, localContext);
                message = evaluator.getMessage(template, transaction, localContext);
            } catch (Exception exception) {
                ErrorHelper.show(exception);
            }
        }
        if (subject == null) {
            String key = isPayment() ? "customer.payment.pp.mail.paymentSubject"
                                     : "customer.payment.pp.mail.refundSubject";
            subject = Messages.format(key, processor.getName());
        }
        if (message == null) {
            String description = transaction.getDescription();
            if (description == null) {
                description = "";
            }
            message = Messages.format("customer.payment.pp.mail.body", processor.getName(), description, url);
        }
        mailEditor.setSubject(subject);
        mailEditor.setMessage(message);
    }

    /**
     * Send an SMS containing the transaction URL to the customer.
     */
    private void onSMS() {
        List<Contact> contacts = ContactHelper.getSMSContacts(context.getContext().getCustomer());
        SMSDialog dialog = new SMSDialog(contacts, context.getContext(), context.getHelpContext());
        dialog.show();
        dialog.setMessage(getSMSMessage());
        // set the message after the dialog is shown, so if the template cannot be evaluated the error will
        // be displayed above the dialog
    }

    /**
     * Returns SMS text including the payment processor link.
     *
     * @return the text
     */
    private String getSMSMessage() {
        String result = null;
        Entity template = getSMSTemplate();
        if (template != null) {
            try {
                PaymentProcessorSMSEvaluator evaluator = new PaymentProcessorSMSEvaluator(
                        context.getContext().getPractice(), ServiceHelper.getBean(SMSTemplateEvaluator.class), service,
                        ServiceHelper.getLookupService());
                result = evaluator.evaluate(template, transaction);
            } catch (Exception exception) {
                ErrorHelper.show(exception);
            }
        }
        if (result == null) {
            String description = transaction.getDescription();
            if (description == null) {
                description = "";
            }
            result = Messages.format("customer.payment.pp.sms", processor.getName(), description, url);
        }
        return result;
    }

    /**
     * Returns the payment or refund email template.
     *
     * @return the template, or {@code null} if none is found
     */
    private EmailTemplate getEmailTemplate() {
        ContextDocumentTemplateLocator locator = new ContextDocumentTemplateLocator(transaction, context.getContext());
        DocumentTemplate template = locator.getTemplate();
        return (template != null) ? template.getEmailTemplate() : null;
    }

    /**
     * Returns the payment or refund SMS template.
     *
     * @return the template, or {@code null} if none is found
     */
    private Entity getSMSTemplate() {
        ContextDocumentTemplateLocator locator = new ContextDocumentTemplateLocator(transaction, context.getContext());
        DocumentTemplate template = locator.getTemplate();
        return (template != null) ? template.getSMSTemplate() : null;
    }

    /**
     * Determines if the transaction is a payment.
     *
     * @return {@code true} if it's a payment, {@code false} if it's a refund
     */
    private boolean isPayment() {
        return transaction.isA(PaymentProcessorArchetypes.PAYMENT);
    }
}