/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.account;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.archetype.rules.finance.statement.StatementRules;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Predicates;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.object.Relationship;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.user.User;
import org.openvpms.web.component.app.LocalContext;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.dialog.ErrorDialog;
import org.openvpms.web.echo.dialog.PopupDialogListener;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.payment.CustomerPaymentEditDialog;
import org.openvpms.web.workspace.customer.payment.CustomerPaymentEditor;

import java.util.Date;
import java.util.List;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;

/**
 * Reverses customer debit and credit acts.
 *
 * @author Tim Anderson
 */
public class Reverser {

    /**
     * Listener used to notify of successful reversal.
     */
    public interface Listener {

        /**
         * Invoked when reversal completes.
         *
         * @param reversal the reversal
         */
        void reversed(FinancialAct reversal);
    }

    /**
     * The practice.
     */
    private final Party practice;

    /**
     * The practice location.
     */
    private final Party location;

    /**
     * The till to use when performing interactive reversals. May be {@code null}
     */
    private final Entity till;

    /**
     * The till to use when performing EFT reversals. May be {@code null}
     */
    private final Entity terminal;

    /**
     * The logged-in user.
     */
    private final User user;

    /**
     * The help context.
     */
    private final HelpContext help;

    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * The customer account rules.
     */
    private final CustomerAccountRules rules;

    /**
     * Constructs a {@link Reverser}.
     *
     * @param practice the practice
     * @param location the current practice location, used to determine EFT terminals to use
     * @param till     the till to use, when performing interactive reversals. May be {@code null}
     * @param terminal the terminal to use, when performing EFT reversals. May be {@code null}
     * @param user     the logged-in user
     * @param help     the help context
     */
    public Reverser(Party practice, Party location, Entity till, Entity terminal, User user, HelpContext help) {
        this.practice = practice;
        this.location = location;
        this.till = till;
        this.terminal = terminal;
        this.user = user;
        this.help = help;
        service = ServiceHelper.getArchetypeService();
        rules = ServiceHelper.getBean(CustomerAccountRules.class);
    }

    /**
     * Reverse a debit or credit act.
     *
     * @param act      the act to reverse. Must be <em>POSTED</em>
     * @param listener the listener to notify on successful reversal
     */
    public void reverse(FinancialAct act, Listener listener) {
        reverse(act, null, listener);
    }

    /**
     * Reverse a debit or credit act.
     *
     * @param act         the act to reverse. Must be <em>POSTED</em>
     * @param tillBalance the till balance to add the reversal to. Only applies to payments and refunds.
     *                    May be {@code null}
     * @param listener    the listener to notify on successful reversal
     */
    public void reverse(FinancialAct act, FinancialAct tillBalance, Listener listener) {
        if (!FinancialActStatus.POSTED.equals(act.getStatus())) {
            throw new IllegalArgumentException("Argument 'act' is not POSTED");
        }
        if (canReverse(act)) {
            if (rules.hasApprovedEFTPOSTransaction(act)) {
                reverseWithPaymentEFTItem(act, listener);
            } else if (hasPaymentProcessorItem(act)) {
                reverseWithPaymentProcessorItem(act, listener);
            } else {
                promptReverse(act, tillBalance, listener);
            }
        }
    }

    /**
     * Creates a new payment editor.
     *
     * @param act     the act to edit
     * @param context the layout context
     * @return a new editor
     */
    protected CustomerPaymentEditor getPaymentEditor(FinancialAct act, DefaultLayoutContext context) {
        return new CustomerPaymentEditor(act, null, context);
    }

    /**
     * Prompts to reverse an act.
     * <p/>
     * The reversal will be <em>POSTED</em>.
     *
     * @param act         the act to reverse. Must be <em>POSTED</em>
     * @param tillBalance the till balance to add the reversal to. Only applies to payments and refunds.
     *                    May be {@code null}
     * @param listener    the listener to notify on successful reversal
     */
    private void promptReverse(FinancialAct act, FinancialAct tillBalance, Listener listener) {
        String name = getDisplayName(act);
        String title = Messages.format("customer.account.reverse.title", name);
        String message = Messages.format("customer.account.reverse.message", name);
        String notes = Messages.format("customer.account.reverse.notes", getDisplayName(act), act.getId());
        String reference = Long.toString(act.getId());

        boolean canHide = canHideReversal(act);
        ReverseConfirmationDialog dialog = new ReverseConfirmationDialog(title, message, help, notes, reference,
                                                                         canHide);
        dialog.addWindowPaneListener(new PopupDialogListener() {
            @Override
            public void onOK() {
                String reversalNotes = dialog.getNotes();
                if (StringUtils.isEmpty(reversalNotes)) {
                    reversalNotes = notes;
                }
                String reversalRef = dialog.getReference();
                if (StringUtils.isEmpty(reversalRef)) {
                    reversalRef = reference;
                }
                FinancialAct reversal = reverse(act, reversalNotes, reversalRef, dialog.getHide(), tillBalance);
                if (reversal != null) {
                    listener.reversed(reversal);
                }
            }
        });
        dialog.show();
    }

    /**
     * Prompts to reverse an act with an <em>act.customerAccountPaymentEFT</em> or
     * <em>act.customerAccountRefundEFT</em>. This is only successful for payments, refunds aren't supported.<p/>
     * For payments, an edit dialog will be displayed.
     *
     * @param act      the act to reverse. Must be <em>POSTED</em>
     * @param listener the listener to notify on successful reversal
     */
    private void reverseWithPaymentEFTItem(FinancialAct act, Listener listener) {
        if (!act.isA(PAYMENT)) {
            showCannotReverseDialog(act, "customer.account.reverse.refundEFT");
        } else {
            doInteractiveReversal(act, listener);
        }
    }

    /**
     * Reverses an act and edits the reversal.
     *
     * @param act      the act to reverse
     * @param listener the listener to notify on completion
     */
    private void doInteractiveReversal(FinancialAct act, Listener listener) {
        String notes = Messages.format("customer.account.reverse.notes", getDisplayName(act), act.getId());
        FinancialAct reversal = rules.createInProgressReversal(act, notes);
        editReversal(reversal, listener);
    }

    /**
     * Edits an IN_PROGRESS reversal.
     *
     * @param reversal   the reversal to edit
     * @param listener the listener to notify on completion
     */
    private void editReversal(FinancialAct reversal, Listener listener) {
        LocalContext context = new LocalContext();
        context.setPractice(practice);
        context.setLocation(location);
        context.setTill(till);
        context.setTerminal(terminal);
        context.setUser(user);
        DefaultLayoutContext layoutContext = new DefaultLayoutContext(context, help);
        layoutContext.setEdit(true);
        CustomerPaymentEditor editor = getPaymentEditor(reversal, layoutContext);
        editor.setStatus(FinancialActStatus.POSTED);
        editor.setTill(till);

        editor.makeSaveableAndPostOnCompletion();
        editor.getItems().editFirst();

        CustomerPaymentEditDialog dialog = new CustomerPaymentEditDialog(editor, context);
        dialog.addWindowPaneListener(new PopupDialogListener() {
            @Override
            public void onOK() {
                listener.reversed(reversal);
            }
        });
        dialog.show();
    }

    /**
     * Displays a dialog indicating that an act cannot be reversed.
     *
     * @param act the act
     * @param key the message resource bundle key
     */
    private void showCannotReverseDialog(FinancialAct act, String key) {
        String displayName = getDisplayName(act);
        ErrorDialog.newDialog()
                .titleKey("customer.account.reverse.title", displayName)
                .messageKey(key, displayName, getDisplayName(PAYMENT))
                .show();
    }

    /**
     * Prompts to reverse an act with an <em>act.customerAccountPaymentPP</em> or
     * <em>act.customerAccountRefundPP</em>. This is only successful for payments, refunds aren't supported.<p/>
     * For payments, an edit dialog will be displayed.
     *
     * @param act      the act to reverse. Must be <em>POSTED</em>
     * @param listener the listener to notify on successful reversal
     */
    private void reverseWithPaymentProcessorItem(FinancialAct act, Listener listener) {
        if (!act.isA(PAYMENT)) {
            showCannotReverseDialog(act, "customer.account.reverse.refundPP");
        } else {
            doInteractiveReversal(act, listener);
        }
    }

    /**
     * Determines if an act can be reversed. If not, displays an error dialog.
     *
     * @param act the act
     * @return {@code true} if it can be reversed, otherwise {@code false}
     */
    private boolean canReverse(FinancialAct act) {
        boolean reversed = rules.isReversed(act);
        if (reversed) {
            IMObjectBean bean = service.getBean(act);
            List<ActRelationship> reversal = bean.getValues("reversal", ActRelationship.class);
            if (!reversal.isEmpty()) {
                Reference target = reversal.get(0).getTarget();
                String reversalDisplayName = getDisplayName(target.getArchetype());
                String displayName = getDisplayName(act);
                ErrorDialog.newDialog()
                        .titleKey("customer.account.reverse.title", displayName)
                        .messageKey("customer.account.reversed.message", displayName,
                                    reversalDisplayName, target.getId())
                        .show();
            }
        }
        return !reversed;
    }

    /**
     * Determines if an act has a payment processor item.
     *
     * @param act the act
     * @return {@code true} if it has a payment processor item, otherwise {@code false}
     */
    private boolean hasPaymentProcessorItem(FinancialAct act) {
        boolean result = false;
        if (act.isA(PAYMENT, REFUND)) {
            IMObjectBean bean = service.getBean(act);
            List<Relationship> relationships = bean.getValues("items", Relationship.class,
                                                              Predicates.targetIsA(PAYMENT_PP, REFUND_PP));
            result = !relationships.isEmpty();
        }
        return result;
    }

    /**
     * Helper to return the display name of an archetype.
     *
     * @param archetype the archetype
     * @return the display name for the archetype
     */
    private String getDisplayName(String archetype) {
        return DescriptorHelper.getDisplayName(archetype, service);
    }

    /**
     * Helper to return the display name of an act.
     *
     * @param act the act
     * @return the display name for the act
     */
    private String getDisplayName(FinancialAct act) {
        return DescriptorHelper.getDisplayName(act, service);
    }

    /**
     * Reverse a debit or credit act.
     *
     * @param act         the act to reverse
     * @param notes       the reversal notes
     * @param reference   the reference
     * @param hide        if {@code true} flag the transaction and its reversal as hidden, so they don't appear in the
     *                    statement
     * @param tillBalance the till balance to add the reversal to. Only applies to payments and refunds.
     *                    May be {@code null}
     * @return the reversal, or {@code null} if it was unsuccessful
     */
    private FinancialAct reverse(FinancialAct act, String notes, String reference, boolean hide,
                                 FinancialAct tillBalance) {
        FinancialAct reversal = null;
        try {
            reversal = rules.reverse(act, new Date(), notes, reference, hide, tillBalance);
        } catch (Exception exception) {
            String title = Messages.format("customer.account.reverse.failed", getDisplayName(act));
            ErrorHelper.show(title, exception);
        }
        return reversal;
    }

    /**
     * Determines if a reversal can be hidden in the customer statement.
     *
     * @param act the act to reverse
     * @return {@code true} if the reversal can be hidden
     */
    private boolean canHideReversal(FinancialAct act) {
        if (!rules.isHidden(act)) {
            StatementRules statementRules = new StatementRules(practice, service, rules);
            IMObjectBean bean = service.getBean(act);
            Party customer = bean.getTarget("customer", Party.class);
            return customer != null && !statementRules.hasStatement(customer, act.getActivityStartTime());
        }
        return false;
    }
}
