/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.admin.template.paymentprocessor;

import nextapp.echo2.app.Component;
import org.openvpms.archetype.rules.customer.CustomerArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.rules.practice.PracticeArchetypes;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.customer.CustomerReferenceEditor;
import org.openvpms.web.component.im.edit.DefaultIMObjectReferenceEditor;
import org.openvpms.web.component.im.edit.IMObjectReferenceEditor;
import org.openvpms.web.component.im.layout.ComponentGrid;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.sms.SMSTemplateEvaluator;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.SimpleProperty;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.admin.template.SMSTemplateSampler;

import java.math.BigDecimal;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;

/**
 * A component to test the expression evaluation of an <em>entity.documentTemplateSMSPaymentProcessor</em>.
 *
 * @author Tim Anderson
 */
public class SMSPaymentProcessorTemplateSampler extends SMSTemplateSampler {

    /**
     * The template evaluator.
     */
    private final PaymentProcessorSMSEvaluator evaluator;

    /**
     * The customer to test against.
     */
    private final Property customer;

    /**
     * The location to test against.
     */
    private final Property location;

    /**
     * The payment processor to test against.
     */
    private final Property processor;

    /**
     * The description to test against.
     */
    private final Property description;

    /**
     * The amount to test against.
     */
    private final Property amount;

    /**
     * The URL to test against.
     */
    private final Property url;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * Constructs an {@link SMSPaymentProcessorTemplateSampler}.
     *
     * @param template      the template
     * @param layoutContext the layout context
     */
    public SMSPaymentProcessorTemplateSampler(Entity template, LayoutContext layoutContext) {
        super(template, layoutContext);
        service = ServiceHelper.getArchetypeService();
        Context context = layoutContext.getContext();
        evaluator = new PaymentProcessorSMSEvaluator(context.getPractice(),
                                                     ServiceHelper.getBean(SMSTemplateEvaluator.class), service,
                                                     ServiceHelper.getLookupService());

        ModifiableListener listener = modifiable -> evaluate();
        customer = SimpleProperty.newProperty()
                .name("customer")
                .type(Reference.class)
                .value(context.getCustomer() != null ? context.getCustomer().getObjectReference() : null)
                .archetypeRange(CustomerArchetypes.PERSON)
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "customer", service))
                .listener(listener)
                .required()
                .build();
        location = SimpleProperty.newProperty()
                .name("location")
                .type(Reference.class)
                .value(context.getLocation() != null ? context.getLocation().getObjectReference() : null)
                .archetypeRange(PracticeArchetypes.LOCATION)
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "location", service))
                .listener(listener)
                .required()
                .build();
        processor = SimpleProperty.newProperty()
                .name("paymentProcessor")
                .type(Reference.class)
                .archetypeRange(PaymentProcessorArchetypes.PAYMENT_PROCESSORS)
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "paymentProcessor", service))
                .listener(listener)
                .required()
                .build();
        description = SimpleProperty.newProperty()
                .name("description")
                .value("deposit for surgery")
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "description", service))
                .listener(listener)
                .required()
                .build();
        amount = SimpleProperty.newProperty()
                .name("amount")
                .value(new BigDecimal("100.00"))
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "amount", service))
                .listener(listener)
                .required()
                .build();
        url = SimpleProperty.newProperty()
                .name("url")
                .value("https://testpaymentprocessor.com/payment/123456789")
                .displayName(DescriptorHelper.getDisplayName(PAYMENT_PP, "url", service))
                .listener(listener)
                .required()
                .build();
    }

    /**
     * Lays out the editable fields in a grid.
     *
     * @param grid    the grid
     * @param group   the focus group
     * @param context the layout context
     */
    @Override
    protected void layoutFields(ComponentGrid grid, FocusGroup group, LayoutContext context) {
        CustomerReferenceEditor customerSelector = new CustomerReferenceEditor(customer, null, context);
        IMObjectReferenceEditor<Party> locationSelector
                = new DefaultIMObjectReferenceEditor<>(location, null, context);
        IMObjectReferenceEditor<Party> processorSelector =
                new DefaultIMObjectReferenceEditor<>(processor, null, context);
        Component descriptionComponent = context.getComponentFactory().create(description);
        Component amountComponent = context.getComponentFactory().create(amount);
        Component urlComponent = context.getComponentFactory().create(url);

        grid.add(LabelFactory.text(customer.getDisplayName()), customerSelector.getComponent(),
                 LabelFactory.text(location.getDisplayName()), locationSelector.getComponent());
        grid.add(LabelFactory.text(processor.getDisplayName()), processorSelector.getComponent(),
                 LabelFactory.text(description.getDisplayName()), descriptionComponent);
        grid.add(LabelFactory.text(amount.getDisplayName()), amountComponent,
                 LabelFactory.text(url.getDisplayName()), urlComponent);
        group.add(customerSelector.getFocusGroup());
        group.add(locationSelector.getFocusGroup());
        group.add(processorSelector.getFocusGroup());
        group.add(descriptionComponent);
        group.add(amountComponent);
        group.add(urlComponent);
    }

    /**
     * Evaluates the template.
     *
     * @param template the template
     * @param context  the context
     * @return the result of the evaluation. May be {@code null}
     */
    @Override
    protected String evaluate(Entity template, Context context) {
        FinancialAct act = service.create(PaymentProcessorArchetypes.PAYMENT, FinancialAct.class);
        IMObjectBean bean = service.getBean(act);
        bean.setTarget("customer", customer.getReference());
        bean.setTarget("location", location.getReference());
        bean.setTarget("paymentProcessor", processor.getReference());
        bean.setValue("description", description.getString());
        bean.setValue("url", url.getString());
        act.setTotal(amount.getBigDecimal(BigDecimal.ZERO));
        return evaluator.evaluate(template, act);
    }
}