/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.admin.job;

import org.openvpms.component.business.service.singleton.SingletonServiceImpl;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.user.User;
import org.openvpms.web.component.im.edit.AbstractIMObjectEditor;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.property.ValidatorError;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.quartz.CronExpression;

import java.text.ParseException;
import java.util.regex.Pattern;

/**
 * Editor for <em>entity.job*</em> archetypes.
 *
 * @author Tim Anderson
 */
public abstract class AbstractJobConfigurationEditor extends AbstractIMObjectEditor {

    /**
     * The bean.
     */
    private final IMObjectBean bean;

    /**
     * The 'Run As' node name.
     */
    private static final String RUN_AS = "runAs";

    /**
     * The notify node name.
     */
    private static final String NOTIFY = "notify";

    /**
     * Constructs an {@link AbstractJobConfigurationEditor}.
     *
     * @param object        the object to edit
     * @param parent        the parent object. May be {@code null}
     * @param layoutContext the layout context
     */
    public AbstractJobConfigurationEditor(Entity object, IMObject parent, LayoutContext layoutContext) {
        super(object, parent, layoutContext);
        bean = getBean(object);
        if (object.isNew()) {
            User user = layoutContext.getContext().getUser();
            if (user != null) {
                initRunAs(user);
                initNotify(user);
            }
        }
    }

    /**
     * Validates the object.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        boolean valid = super.doValidation(validator);
        if (valid) {
            valid = validateMinutes(validator) && validateHours(validator) && validateDayOfMonth(validator)
                    && validateMonth(validator) && validateDayOfWeek(validator)
                    && validateRunAs(validator) && validateNotify(validator);
            if (valid) {
                // try and parse the expression
                Property property = getProperty("expression");
                String expression = property.getString();
                try {
                    new CronExpression(expression);
                } catch (ParseException exception) {
                    valid = false;
                    validator.add(property, new ValidatorError(property, exception.getMessage()));
                }
            }
        }
        return valid;
    }

    /**
     * Helper to verifies that there is only one active job at a time.
     *
     * @param validator the validator
     * @return {@code true} if the job is valid
     */
    protected boolean checkSingleton(Validator validator) {
        boolean valid = true;
        IMObject object = getObject();
        if (object.isActive()) {
            SingletonServiceImpl service = ServiceHelper.getBean(SingletonServiceImpl.class);
            if (service.exists(object.getArchetype(), object.getObjectReference())) {
                valid = false;
                validator.add(this, new ValidatorError(Messages.format("job.exists", getDisplayName())));
            }
        }
        return valid;
    }

    /**
     * Initialises the runAs node from the specified user.
     *
     * @param user the user
     */
    protected void initRunAs(User user) {
        if (getProperty(RUN_AS) != null) {
            bean.setTarget(RUN_AS, user);
        }
    }

    /**
     * Initialises the notify node from the specified user.
     *
     * @param user the user
     */
    protected void initNotify(User user) {
        if (getProperty(NOTIFY) != null) {
            bean.setTarget(NOTIFY, user);
        }
    }

    /**
     * Validates the minutes property.
     *
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateMinutes(Validator validator) {
        return validateProperty("minutes", CronHelper.MINUTES, validator);
    }

    /**
     * Validates the hours property.
     *
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateHours(Validator validator) {
        return validateProperty("hours", CronHelper.HOURS, validator);
    }

    /**
     * Validates the dayOfMonth property.
     *
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateDayOfMonth(Validator validator) {
        return validateProperty("dayOfMonth", CronHelper.DAY_OF_MONTH, validator);
    }

    /**
     * Validates the month property.
     *
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateMonth(Validator validator) {
        return validateProperty("month", CronHelper.MONTH, validator);
    }

    /**
     * Validates the dayOfWeek property.
     *
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateDayOfWeek(Validator validator) {
        return validateProperty("dayOfWeek", CronHelper.DAY_OF_WEEK, validator);
    }

    /**
     * Validates a property.
     *
     * @param name      the property name
     * @param pattern   the pattern to validate against
     * @param validator the validator
     * @return {@code true} if the property is valid
     */
    private boolean validateProperty(String name, Pattern pattern, Validator validator) {
        Property property = getProperty(name);
        String value = property.getString();
        if (value == null) {
            value = "";
        }
        boolean valid = pattern.matcher(value).matches();
        if (!valid) {
            validator.add(property, new ValidatorError(property, Messages.format("job.property.invalid", value)));
        }
        return valid;
    }

    /**
     * Ensures that any 'run as' user is active.
     *
     * @param validator the validator
     * @return {@code true} if there is no user, or the user is active
     */
    private boolean validateRunAs(Validator validator) {
        return validateUser(RUN_AS, validator);
    }

    /**
     * Ensures that any 'notify' user is active.
     *
     * @param validator the validator
     * @return {@code true} if there is no user, or the user is active
     */
    private boolean validateNotify(Validator validator) {
        return validateUser(NOTIFY, validator);
    }

    /**
     * Ensures that any user or group associated with the named property is active.
     *
     * @param name      the property name
     * @param validator the validator
     * @return {@code true} if there is no property/user or there is, and the user is active
     */
    private boolean validateUser(String name, Validator validator) {
        boolean valid = true;
        Property property = getProperty(name);
        if (property != null) {
            Entity userOrGroup = (Entity) getObject(bean.getTargetRef(name));
            if (userOrGroup != null && !userOrGroup.isActive()) {
                validator.add(property, new ValidatorError(Messages.format("admin.job.user.inactive",
                                                                           property.getDisplayName())));
                valid = false;
            }
        }
        return valid;
    }

}
