/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.workflow.otc;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.PracticeMailContext;
import org.openvpms.web.component.im.util.UserHelper;
import org.openvpms.web.component.workflow.DefaultTaskContext;
import org.openvpms.web.component.workflow.EditIMObjectTask;
import org.openvpms.web.component.workflow.PrintActTask;
import org.openvpms.web.component.workflow.PrintIMObjectTask;
import org.openvpms.web.component.workflow.SynchronousTask;
import org.openvpms.web.component.workflow.TaskContext;
import org.openvpms.web.component.workflow.WorkflowImpl;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.workflow.WorkflowException;
import org.openvpms.web.workspace.workflow.i18n.WorkflowMessages;

import java.util.Arrays;


/**
 * Over-the-counter workflow.
 *
 * @author Tim Anderson
 */
public class OverTheCounterWorkflow extends WorkflowImpl {

    /**
     * The initial context.
     */
    private final TaskContext initial;

    /**
     * Constructs an {@link OverTheCounterWorkflow}.
     *
     * @param parent the parent context
     * @param help   the help context
     * @throws ArchetypeServiceException for any archetype service error
     * @throws WorkflowException         for any workflow error
     */
    public OverTheCounterWorkflow(final Context parent, HelpContext help) {
        super(help);
        if (parent.getPractice() == null) {
            throw new WorkflowException(WorkflowMessages.contextHasNoPractice());
        }
        initial = new DefaultTaskContext(null, help);
        Party location = parent.getLocation();
        if (location == null) {
            throw new WorkflowException(WorkflowMessages.contextHasNoLocation());
        }
        IMObjectBean bean = getBean(location);
        Party otc = bean.getTarget("OTC", Party.class);
        if (otc == null) {
            throw new WorkflowException(WorkflowMessages.locationHasNoOTCCustomer(location));
        }
        initial.setCustomer(otc);
        initial.setTill(parent.getTill());
        initial.setTerminal(parent.getTerminal());
        initial.setLocation(parent.getLocation());
        initial.setPractice(parent.getPractice());
        initial.setUser(parent.getUser());

        EditIMObjectTask charge = createChargeTask();
        charge.setDeleteOnCancelOrSkip(true);
        addTask(charge);
        EditIMObjectTask payment = createPaymentTask();
        payment.setDeleteOnCancelOrSkip(true);
        addTask(payment);
        addTask(new SynchronousTask() {
            @Override
            public void execute(TaskContext context) {
                Act charge = (Act) context.getObject(CustomerAccountArchetypes.COUNTER);
                Act payment = (Act) context.getObject(CustomerAccountArchetypes.PAYMENT);
                charge.setStatus(ActStatus.POSTED);
                payment.setStatus(ActStatus.POSTED);
                ServiceHelper.getArchetypeService().save(Arrays.asList(charge, payment));
            }
        });

        // optionally select and print the counter charge
        PrintIMObjectTask printTask = createPrintTask(parent);
        printTask.setRequired(false);
        addTask(printTask);

        // add a task to update the global context at the end of the workflow
        addTask(new SynchronousTask() {
            public void execute(TaskContext context) {
                parent.setTill(context.getTill());
                parent.setTerminal(context.getTerminal());
                if (!UserHelper.useLoggedInClinician(context)) {
                    parent.setClinician(context.getClinician());
                }
            }
        });

        setBreakOnSkip(true);
    }

    /**
     * Starts the workflow.
     */
    @Override
    public void start() {
        super.start(initial);
    }

    /**
     * Creates a task to edit the charge.
     *
     * @return a new task
     */
    protected EditIMObjectTask createChargeTask() {
        return new OTCChargeTask();
    }

    /**
     * Creates a task to edit the payment.
     *
     * @return a new task
     */
    protected EditIMObjectTask createPaymentTask() {
        return new OTCPaymentTask();
    }

    /**
     * Creates a task to print the charge.
     *
     * @param parent the parent context
     * @return a new task
     */
    protected PrintIMObjectTask createPrintTask(Context parent) {
        return new PrintActTask(CustomerAccountArchetypes.COUNTER, new PracticeMailContext(parent));
    }

}
