/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.report;

import nextapp.echo2.app.Component;
import org.openvpms.archetype.rules.doc.DocumentTemplate;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.user.User;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.component.system.common.util.Variables;
import org.openvpms.report.ReportFactory;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.ReloadingContext;
import org.openvpms.web.component.im.doc.FileNameFormatter;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.print.PrinterContext;
import org.openvpms.web.component.im.print.PrinterContextFactory;
import org.openvpms.web.component.im.query.Browser;
import org.openvpms.web.component.im.query.BrowserListener;
import org.openvpms.web.component.im.query.DefaultIMObjectTableBrowser;
import org.openvpms.web.component.im.table.DefaultDescriptorTableModel;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.macro.MacroVariables;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.reporting.AbstractReportingWorkspace;

import javax.sql.DataSource;
import java.util.List;


/**
 * Reporting workspace.
 *
 * @author Tim Anderson
 */
public class ReportingWorkspace extends AbstractReportingWorkspace<Entity> {

    /**
     * The current user. May be {@code null}.
     */
    private final User user;

    /**
     * The entity browser.
     */
    private Browser<Entity> browser;

    /**
     * Run button identifier.
     */
    private static final String RUN_ID = "run";


    /**
     * Constructs a {@code ReportingWorkspace}.
     *
     * @param context     the context
     * @param mailContext the mail context
     */
    public ReportingWorkspace(Context context, MailContext mailContext) {
        super("reporting.report", Entity.class, context, mailContext);
        user = getContext().getUser();
    }

    /**
     * Lays out the components.
     *
     * @param container the container
     * @param group     the focus group
     */
    @Override
    protected void doLayout(Component container, FocusGroup group) {
        if (user != null) {
            layoutWorkspace(user, container);
        }
    }

    /**
     * Determines if the workspace should be refreshed. This implementation
     * returns true if the current user has changed.
     *
     * @return {@code true} if the workspace should be refreshed, otherwise {@code false}
     */
    @Override
    protected boolean refreshWorkspace() {
        User user = getContext().getUser();
        user = IMObjectHelper.reload(user);
        return !IMObjectHelper.isSame(this.user, user);
    }

    /**
     * Lays out the workspace.
     *
     * @param user      the user
     * @param container the container
     */
    protected void layoutWorkspace(User user, Component container) {
        ReportQuery query = createQuery(user);
        browser = createBrowser(query);
        browser.addBrowserListener(new BrowserListener<Entity>() {
            public void query() {
                selectFirst();
            }

            public void selected(Entity object) {
                setObject(object);
            }

            public void browsed(Entity object) {
                setObject(object);
            }
        });
        container.add(browser.getComponent());
        if (!query.isAuto()) {
            browser.query();
        }
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button set
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        buttons.add(RUN_ID, this::onRun);
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        super.enableButtons(buttons, enable);
        buttons.setEnabled(RUN_ID, enable);
    }

    /**
     * Invoked when the run button is pressed. Runs the selected report.
     */
    protected void onRun() {
        Entity entity = getObject();
        if (entity != null) {
            IArchetypeService service = ServiceHelper.getArchetypeService();
            LookupService lookups = ServiceHelper.getLookupService();
            DocumentTemplate template = new DocumentTemplate(entity, service);
            Context context = getContext();
            ReportFactory factory = ServiceHelper.getBean(ReportFactory.class);
            FileNameFormatter formatter = ServiceHelper.getBean(FileNameFormatter.class);
            DataSource dataSource = ServiceHelper.getBean("reportingDataSource", DataSource.class);
            PrinterContext printerContext = ServiceHelper.getBean(PrinterContextFactory.class).create();
            SQLReportPrinter printer = new SQLReportPrinter(template, context, factory, formatter, dataSource,
                                                            printerContext);
            HelpContext help = getHelpContext().subtopic("run");
            Variables variables = new MacroVariables(new ReloadingContext(context), service, lookups);
            InteractiveSQLReportPrinter iPrinter = new InteractiveSQLReportPrinter(
                    printer, context, getMailContext(), help, variables);
            iPrinter.print();
        }
    }

    /**
     * Creates the report browser.
     *
     * @param query the entity query
     * @return a new act browser
     */
    private Browser<Entity> createBrowser(ReportQuery query) {
        DefaultLayoutContext context = new DefaultLayoutContext(getContext(), getHelpContext());
        DefaultDescriptorTableModel<Entity> model = new DefaultDescriptorTableModel<>(
                query.getShortNames(), context, "id", "name", "description", "reportType");
        return new DefaultIMObjectTableBrowser<>(query, model, context);
    }

    /**
     * Creates a new query.
     *
     * @param user the user to query
     * @return a new query
     */
    private ReportQuery createQuery(User user) {
        return new ReportQuery(user);
    }

    /**
     * Selects the first available report.
     */
    private void selectFirst() {
        List<Entity> objects = browser.getObjects();
        if (!objects.isEmpty()) {
            Entity current = objects.get(0);
            browser.setSelected(current);
            setObject(current);
        } else {
            setObject(null);
        }
    }

}
