/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.payment;

import nextapp.echo2.app.Button;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.LocalContext;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.ActActions;
import org.openvpms.web.component.im.edit.ViewActActions;
import org.openvpms.web.component.im.query.QueryBrowser;
import org.openvpms.web.component.im.view.IMObjectViewerDialog;
import org.openvpms.web.component.im.view.Selection;
import org.openvpms.web.component.im.view.ViewResultSetDialog;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.component.workspace.ResultSetCRUDWindow;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.dialog.InformationDialog;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.workspace.customer.CustomerMailContext;

import java.util.ArrayList;
import java.util.List;

/**
 * CRUD window to view 3rd-party payment and refund transactions.
 *
 * @author Tim Anderson
 */
public class TransactionCRUDWindow extends ResultSetCRUDWindow<FinancialAct> {

    /**
     * The  query browser.
     */
    private final QueryBrowser<FinancialAct> browser;

    /**
     * Source button identifier.
     */
    private static final String SOURCE_ID = "button.source";

    /**
     * Constructs a {@link TransactionCRUDWindow}.
     *
     * @param archetypes the transaction archetypes
     * @param browser the browser
     * @param context    the context
     * @param help       the help context
     */
    public TransactionCRUDWindow(Archetypes<FinancialAct> archetypes, QueryBrowser<FinancialAct> browser,
                                 Context context, HelpContext help) {
        this(archetypes, ViewActActions.getInstance(), browser, context, help);
    }

    /**
     * Constructs a {@link TransactionCRUDWindow}.
     *
     * @param archetypes the transaction archetypes
     * @param actions    the actions
     * @param browser    the browser
     * @param context    the context
     * @param help       the help context
     */
    protected TransactionCRUDWindow(Archetypes<FinancialAct> archetypes, ActActions<FinancialAct> actions,
                                    QueryBrowser<FinancialAct> browser, Context context, HelpContext help) {
        super(archetypes, actions, browser.getQuery(), null, context, help);
        this.browser = browser;
    }

    /**
     * Returns the customer associated with the selected transaction.
     *
     * @return the customer. May be {@code null}
     */
    public Party getCustomer() {
        Party result = null;
        FinancialAct object = getObject();
        if (object != null) {
            result = getCustomer(object);
        }
        return result;
    }

    /**
     * Views the selected object.
     */
    @Override
    public void view() {
        setResultSet(browser.getResultSet());
        super.view();
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        buttons.add(createViewButton());
        buttons.add(SOURCE_ID, action(this::viewPaymentOrRefund));
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        buttons.setEnabled(VIEW_ID, enable);
        buttons.setEnabled(SOURCE_ID, enable);
    }

    /**
     * Creates a new {@link ViewResultSetDialog}.
     *
     * @param title  the dialog title
     * @param object the object to view
     * @param edit   if {@code true} display an edit button
     * @return a new dialog
     */
    @Override
    protected ViewResultSetDialog<FinancialAct> createViewResultSetDialog(String title, FinancialAct object,
                                                                          boolean edit) {
        return new SourceViewResultSetDialog(title, object, edit);
    }

    /**
     * Returns the customer associated with an object.
     *
     * @param object the object
     * @return the customer. May be {@code null}
     */
    private Party getCustomer(FinancialAct object) {
        Party result;
        IMObjectBean bean = getBean(object);
        result = bean.getTarget("customer", Party.class);
        return result;
    }

    /**
     * Views the payment/refund associated with a transaction.
     *
     * @param object the transaction
     */
    private void viewPaymentOrRefund(FinancialAct object) {
        List<Selection> selections = new ArrayList<>();
        FinancialAct act = getPaymentOrRefund(object, selections);
        if (act != null) {
            LocalContext context = LocalContext.copy(getContext());
            context.setCustomer(getCustomer());
            MailContext mailContext = new CustomerMailContext(context);
            IMObjectViewerDialog dialog = new IMObjectViewerDialog(context, mailContext, getHelpContext());
            dialog.disableHyperlinks();
            dialog.setObject(act, selections);
            dialog.show();
        } else {
            InformationDialog.newDialog().
                    titleKey("reporting.payment.sourcenotfound")
                    .show();
        }
    }

    /**
     * Returns the parent payment/refund of a payment processor transaction.
     *
     * @param act the transaction
     * @return the parent payment/refund. May be {@code null}
     */
    private FinancialAct getPaymentOrRefund(FinancialAct act, List<Selection> selection) {
        FinancialAct parent = null;
        IMObjectBean transactionBean = getBean(act);
        FinancialAct item = transactionBean.getSource("transaction", FinancialAct.class);
        if (item != null) {
            IMObjectBean itemBean = getBean(item);
            parent = itemBean.getSource("transaction", FinancialAct.class);
            if (parent != null) {
                selection.add(new Selection("items", null));
                selection.add(new Selection(null, itemBean.getObject("transaction", ActRelationship.class)));
                selection.add(new Selection("transactions", null));
                selection.add(new Selection(null, transactionBean.getObject("transaction", ActRelationship.class)));
            }
        }
        return parent;
    }

    /**
     * A {@link ViewResultSetDialog} that enables the source payment/refund to be viewed.
     */
    protected class SourceViewResultSetDialog extends IMObjectViewResultSetDialog {

        public SourceViewResultSetDialog(String title, FinancialAct object, boolean edit) {
            super(title, object, edit);
            ButtonSet buttons = getButtons();
            Button button = buttons.create(SOURCE_ID, this::viewPaymentOrRefund);
            buttons.add(button, 0);
        }

        /**
         * Views the payment/refund of the selected transaction.
         */
        private void viewPaymentOrRefund() {
            FinancialAct selected = getSelected();
            if (selected != null) {
                TransactionCRUDWindow.this.viewPaymentOrRefund(selected);
            }
        }
    }
}
