/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.payment;

import org.openvpms.archetype.rules.prefs.Preferences;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.DefaultContextSwitchListener;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.query.BrowserFactory;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.query.QueryBrowser;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.component.workspace.TabComponent;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.tabpane.ObjectTabPaneModel;
import org.openvpms.web.workspace.reporting.account.AccountActCRUDWindow;
import org.openvpms.web.workspace.reporting.account.AccountReportingWorkspace;
import org.openvpms.web.workspace.reporting.payment.eftpos.EFTPOSCRUDWindow;
import org.openvpms.web.workspace.reporting.payment.eftpos.EFTPOSQuery;
import org.openvpms.web.workspace.reporting.payment.eftpos.EFTPOSTableModel;
import org.openvpms.web.workspace.reporting.payment.paymentprocessor.PaymentProcessorActQuery;
import org.openvpms.web.workspace.reporting.payment.paymentprocessor.PaymentProcessorActTableModel;
import org.openvpms.web.workspace.reporting.payment.paymentprocessor.PaymentProcessorCRUDWindow;


/**
 * Workspace to:
 * <ul>
 *     <li>detail customer payments that are works-in-progress, i.e not POSTED.</li>
 *     <li>query payments</li>
 *     <li>query EFT transactions</li>
 *     <li>query payment processor transactions</li>
 * </ul>
 *
 * @author Tim Anderson
 */
public class PaymentsWorkspace extends AccountReportingWorkspace {

    /**
     * Constructs a {@link PaymentsWorkspace}.
     *
     * @param context     the context
     * @param mailContext the mail context
     * @param preferences the preferences
     */
    public PaymentsWorkspace(Context context, MailContext mailContext, Preferences preferences) {
        super("reporting.payment", context, mailContext, preferences);
    }

    /**
     * Adds tabs to the tabbed pane.
     *
     * @param model the tabbed pane model
     */
    @Override
    protected void addTabs(ObjectTabPaneModel<TabComponent> model) {
        super.addTabs(model);
        addEFTPOSBrowser(model);
        addPaymentProcessorBrowser(model);
    }

    /**
     * Adds a browser to search for EFT transactions.
     *
     * @param model the tab model
     */
    protected void addEFTPOSBrowser(ObjectTabPaneModel<TabComponent> model) {
        HelpContext help = subtopic("search");
        DefaultLayoutContext layoutContext = new DefaultLayoutContext(getContext(), help);
        layoutContext.setContextSwitchListener(DefaultContextSwitchListener.INSTANCE);
        Query<FinancialAct> query = new EFTPOSQuery(layoutContext);
        QueryBrowser<FinancialAct> browser = (QueryBrowser<FinancialAct>) BrowserFactory.create(
                query, null, new EFTPOSTableModel(layoutContext), layoutContext);
        Archetypes<FinancialAct> archetypes = Archetypes.create(query.getShortNames(), FinancialAct.class);
        EFTPOSCRUDWindow window = new EFTPOSCRUDWindow(archetypes, browser, getContext(), help);
        addTab("reporting.eftpos", model, new TransactionTab(browser, window, false));
    }

    /**
     * Adds a browser to search for EFT transactions.
     *
     * @param model the tab model
     */
    protected void addPaymentProcessorBrowser(ObjectTabPaneModel<TabComponent> model) {
        HelpContext help = subtopic("search");
        DefaultLayoutContext layoutContext = new DefaultLayoutContext(getContext(), help);
        layoutContext.setContextSwitchListener(DefaultContextSwitchListener.INSTANCE);
        Query<FinancialAct> query = new PaymentProcessorActQuery(layoutContext);
        QueryBrowser<FinancialAct> browser = (QueryBrowser<FinancialAct>) BrowserFactory.create(
                query, null, new PaymentProcessorActTableModel(layoutContext), layoutContext);
        Archetypes<FinancialAct> archetypes = Archetypes.create(query.getShortNames(), FinancialAct.class);
        TransactionCRUDWindow window = new PaymentProcessorCRUDWindow(archetypes, browser, getContext(), help);
        addTab("reporting.paymentprocessor", model, new TransactionTab(browser, window, false));
    }

    /**
     * Creates a query for work-in-progress acts.
     *
     * @param help the help context
     * @return a new query
     */
    @Override
    protected Query<FinancialAct> createWorkInProgressQuery(HelpContext help) {
        return new WorkInProgressPaymentsQuery(new DefaultLayoutContext(getContext(), help));
    }

    /**
     * Creates a CRUD window for work-in-progress account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    @Override
    protected AccountActCRUDWindow createWorkInProgressCRUDWindow(
            Archetypes<FinancialAct> archetypes, QueryBrowser<FinancialAct> browser, HelpContext help) {
        return new AccountActCRUDWindow(archetypes, browser, "WORK_IN_PROGRESS_PAYMENTS", getContext(), help);
    }

    /**
     * Creates a query for to search acts by location and amount.
     *
     * @param help the help context
     * @return a new query
     */
    @Override
    protected Query<FinancialAct> createSearchQuery(HelpContext help) {
        return new PaymentsQuery(new DefaultLayoutContext(getContext(), help));
    }

    /**
     * Creates a CRUD window to search account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    @Override
    protected AccountActCRUDWindow createSearchCRUDWindow(Archetypes<FinancialAct> archetypes,
                                                          QueryBrowser<FinancialAct> browser, HelpContext help) {
        return new AccountActCRUDWindow(archetypes, browser, "PAYMENTS", getContext(), help);
    }
}