/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.charge;

import org.openvpms.archetype.rules.prefs.Preferences;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.query.QueryBrowser;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.workspace.reporting.account.AccountActCRUDWindow;
import org.openvpms.web.workspace.reporting.account.AccountReportingWorkspace;


/**
 * Workspace to:
 * <ul>
 *     <li>detail customer charges that are works-in-progress, i.e not POSTED.</li>
 *     <li>query charges</li>
 * </ul>
 *
 * @author Tim Anderson
 */
public class ChargesWorkspace extends AccountReportingWorkspace {

    /**
     * Constructs a {@link ChargesWorkspace}.
     *
     * @param context     the context
     * @param mailContext the mail context
     * @param preferences the preferences
     */
    public ChargesWorkspace(Context context, MailContext mailContext, Preferences preferences) {
        super("reporting.charge", context, mailContext, preferences);
    }

    /**
     * Creates a query for work-in-progress acts.
     *
     * @param help the help context
     * @return a new query
     */
    @Override
    protected Query<FinancialAct> createWorkInProgressQuery(HelpContext help) {
        return new IncompleteChargesQuery(new DefaultLayoutContext(getContext(), help));
    }

    /**
     * Creates a CRUD window for work-in-progress account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    @Override
    protected AccountActCRUDWindow createWorkInProgressCRUDWindow(
            Archetypes<FinancialAct> archetypes, QueryBrowser<FinancialAct> browser, HelpContext help) {
        return new IncompleteChargesCRUDWindow(archetypes, browser, getContext(), help);
    }

    /**
     * Creates a query for to search acts by location and amount.
     *
     * @param help the help context
     * @return a new query
     */
    @Override
    protected Query<FinancialAct> createSearchQuery(HelpContext help) {
        return new ChargesQuery(new DefaultLayoutContext(getContext(), help));
    }

    /**
     * Creates a CRUD window to search account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    @Override
    protected AccountActCRUDWindow createSearchCRUDWindow(Archetypes<FinancialAct> archetypes,
                                                          QueryBrowser<FinancialAct> browser, HelpContext help) {
        return new AccountActCRUDWindow(archetypes, browser, "CHARGES", getContext(), help);
    }
}
