/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.account;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.SplitPane;
import org.openvpms.archetype.rules.prefs.Preferences;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.DefaultContextSwitchListener;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.query.Browser;
import org.openvpms.web.component.im.query.BrowserFactory;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.query.QueryBrowser;
import org.openvpms.web.component.im.table.DescriptorTableModel;
import org.openvpms.web.component.im.table.IMObjectTableModel;
import org.openvpms.web.component.mail.MailContext;
import org.openvpms.web.component.workspace.AbstractCRUDWindow;
import org.openvpms.web.component.workspace.BrowserCRUDWindowTab;
import org.openvpms.web.component.workspace.ResultSetCRUDWindow;
import org.openvpms.web.component.workspace.TabComponent;
import org.openvpms.web.component.workspace.TabbedWorkspace;
import org.openvpms.web.echo.factory.SplitPaneFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.tabpane.ObjectTabPaneModel;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.CustomerSummary;
import org.openvpms.web.workspace.patient.summary.CustomerPatientSummaryFactory;
import org.openvpms.web.workspace.reporting.payment.PaymentsWorkspace;
import org.openvpms.web.workspace.reporting.payment.TransactionCRUDWindow;

/**
 * Workspace to:
 * <ul>
 *     <li>detail customer account acts that are works-in-progress, i.e not POSTED.</li>
 *     <li>query account acts</li>
 * </ul>
 *
 * @author Tim Anderson
 */
public abstract class AccountReportingWorkspace extends TabbedWorkspace<FinancialAct> {

    /**
     * The user preferences.
     */
    private final Preferences preferences;

    /**
     * Constructs an {@link AccountReportingWorkspace}.
     *
     * @param id          the workspace id
     * @param context     the context
     * @param mailContext the mail context
     * @param preferences the user preferences
     */
    public AccountReportingWorkspace(String id, Context context, MailContext mailContext, Preferences preferences) {
        super(id, context);
        setMailContext(mailContext);
        this.preferences = preferences;
    }

    /**
     * Renders the workspace summary.
     *
     * @return the component representing the workspace summary, or {@code null} if there is no summary
     */
    @Override
    public Component getSummary() {
        Component result = null;
        Tab tab = (Tab) getSelected();
        Party customer = (tab != null) ? tab.getCustomer() : null;
        if (customer != null) {
            CustomerPatientSummaryFactory factory = ServiceHelper.getBean(CustomerPatientSummaryFactory.class);
            CustomerSummary summary = factory.createCustomerSummary(getContext(), getHelpContext(), preferences);
            result = summary.getSummary(customer);
        }
        return result;
    }

    /**
     * Returns the class type that this operates on.
     *
     * @return the class type that this operates on
     */
    @Override
    protected Class<FinancialAct> getType() {
        return FinancialAct.class;
    }

    /**
     * Adds tabs to the tabbed pane.
     *
     * @param model the tabbed pane model
     */
    @Override
    protected void addTabs(ObjectTabPaneModel<TabComponent> model) {
        addWorkInProgress(model);
        addSearch(model);
    }

    /**
     * Adds a browser for work-in-progress (i.e. not POSTED) acts.
     *
     * @param model the tab model
     */
    protected void addWorkInProgress(ObjectTabPaneModel<TabComponent> model) {
        HelpContext help = subtopic("wip");
        Query<FinancialAct> query = createWorkInProgressQuery(help);

        DefaultLayoutContext layoutContext = new DefaultLayoutContext(getContext(), help);
        layoutContext.setContextSwitchListener(DefaultContextSwitchListener.INSTANCE);
        QueryBrowser<FinancialAct> browser = (QueryBrowser<FinancialAct>) BrowserFactory.create(
                query, null, createTableModel(query.getShortNames(), layoutContext), layoutContext);
        Archetypes<FinancialAct> archetypes = Archetypes.create(query.getShortNames(), FinancialAct.class);
        AccountActCRUDWindow window = createWorkInProgressCRUDWindow(archetypes, browser, help);
        addTab("reporting.account.wip", model, new AccountTab(browser, window));
    }

    /**
     * Creates a query for work-in-progress acts.
     *
     * @param help the help context
     * @return a new query
     */
    protected abstract Query<FinancialAct> createWorkInProgressQuery(HelpContext help);

    /**
     * Creates a CRUD window for work-in-progress account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    protected abstract AccountActCRUDWindow createWorkInProgressCRUDWindow(Archetypes<FinancialAct> archetypes,
                                                                           QueryBrowser<FinancialAct> browser,
                                                                           HelpContext help);

    /**
     * Creates a query for to search acts by location and amount.
     *
     * @param help the help context
     * @return a new query
     */
    protected abstract Query<FinancialAct> createSearchQuery(HelpContext help);

    /**
     * Creates a CRUD window to search account acts.
     *
     * @param archetypes the archetypes being queried
     * @param browser    the browser
     * @param help       the help context
     * @return a new window
     */
    protected abstract AccountActCRUDWindow createSearchCRUDWindow(
            Archetypes<FinancialAct> archetypes, QueryBrowser<FinancialAct> browser, HelpContext help);

    /**
     * Creates a table to display the acts in a browser.
     *
     * @param context the layout context
     * @return a new table model
     */
    protected IMObjectTableModel<FinancialAct> createTableModel(String[] archetypes, LayoutContext context) {
        return new TableModel(archetypes, context);
    }

    /**
     * Adds a browser to search for charges.
     *
     * @param model the tab model
     */
    protected void addSearch(ObjectTabPaneModel<TabComponent> model) {
        HelpContext help = subtopic("search");
        Query<FinancialAct> query = createSearchQuery(help);
        DefaultLayoutContext layoutContext = new DefaultLayoutContext(getContext(), help);
        layoutContext.setContextSwitchListener(DefaultContextSwitchListener.INSTANCE);
        QueryBrowser<FinancialAct> browser = (QueryBrowser<FinancialAct>) BrowserFactory.create(
                query, null, createTableModel(query.getShortNames(), layoutContext), layoutContext);
        Archetypes<FinancialAct> archetypes = Archetypes.create(query.getShortNames(), FinancialAct.class);
        AccountActCRUDWindow window = createSearchCRUDWindow(archetypes, browser, help);
        addTab("reporting.account.search", model, new AccountTab(browser, window, false));
    }


    protected abstract class Tab extends BrowserCRUDWindowTab<FinancialAct> {

        /**
         * Constructs a {@link PaymentsWorkspace.Tab}.
         *
         * @param browser the browser
         * @param window  the window
         */
        public Tab(Browser<FinancialAct> browser, AbstractCRUDWindow<FinancialAct> window) {
            super(browser, window);
        }

        /**
         * Constructs a {@link PaymentsWorkspace.Tab}.
         *
         * @param browser       the browser
         * @param window        the window
         * @param refreshOnShow determines if the browser should be refreshed when the tab is displayed.
         */
        public Tab(Browser<FinancialAct> browser, AbstractCRUDWindow<FinancialAct> window, boolean refreshOnShow) {
            super(browser, window, refreshOnShow);
        }

        /**
         * Invoked when the tab is displayed.
         */
        @Override
        public void show() {
            super.show();
            firePropertyChange(SUMMARY_PROPERTY, null, null);
        }

        /**
         * Returns the tab component.
         *
         * @return the tab component
         */
        @Override
        public Component getComponent() {
            return SplitPaneFactory.create(SplitPane.ORIENTATION_VERTICAL_BOTTOM_TOP, "CRUDWindow",
                                           getWindow().getComponent(), getBrowser().getComponent());
        }

        /**
         * Returns the customer of the selected object.
         *
         * @return the customer, or {@code null} if there is none
         */
        public abstract Party getCustomer();

        /**
         * Selects the current object.
         *
         * @param object the selected object
         */
        @Override
        protected void select(FinancialAct object) {
            boolean refresh = refreshSummary(object);
            super.select(object);
            if (refresh) {
                firePropertyChange(SUMMARY_PROPERTY, null, null);
            }
        }

        /**
         * Refresh the browser.
         *
         * @param object the object to select. May be {@code null}
         */
        @Override
        protected void refreshBrowser(FinancialAct object) {
            boolean refresh = refreshSummary(object);
            super.refreshBrowser(object);
            if (refresh) {
                firePropertyChange(SUMMARY_PROPERTY, null, null);
            }
        }

        /**
         * Invoked when an object is double-clicked.
         * <p/>
         * Implementations can edit the object if editing is supported, or view it.
         */
        @Override
        protected abstract void onDoubleClick();

        /**
         * Determines if the summary should be refreshed.
         *
         * @param object the object to select. May be {@code null}
         * @return {@code true} the current selected tab is the same as this, and {@code object} is different to the
         * existing selection or {@code null}
         */
        private boolean refreshSummary(FinancialAct object) {
            FinancialAct current = getWindow().getObject();
            return getSelected() == this && current != object || object == null;
        }
    }

    protected class AccountTab extends Tab {

        /**
         * Constructs a {@link Tab}.
         *
         * @param browser the browser
         * @param window  the window
         */
        public AccountTab(Browser<FinancialAct> browser, AccountActCRUDWindow window) {
            super(browser, window);
        }

        /**
         * Constructs a {@link Tab}.
         *
         * @param browser       the browser
         * @param window        the window
         * @param refreshOnShow determines if the browser should be refreshed when the tab is displayed.
         */
        public AccountTab(Browser<FinancialAct> browser, ResultSetCRUDWindow<FinancialAct> window,
                          boolean refreshOnShow) {
            super(browser, window, refreshOnShow);
        }

        /**
         * Returns the customer of the selected object.
         *
         * @return the customer, or {@code null} if there is none
         */
        @Override
        public Party getCustomer() {
            return getWindow().getCustomer();
        }

        /**
         * Returns the CRUD window.
         *
         * @return the window
         */
        @Override
        public AccountActCRUDWindow getWindow() {
            return (AccountActCRUDWindow) super.getWindow();
        }

        /**
         * Invoked when an object is double-clicked.
         * <p/>
         * This views the selected object.
         */
        @Override
        protected void onDoubleClick() {
            getWindow().view();
        }
    }

    protected class TransactionTab extends Tab {

        /**
         * Constructs a {@link Tab}.
         *
         * @param browser       the browser
         * @param window        the window
         * @param refreshOnShow determines if the browser should be refreshed when the tab is displayed.
         */
        public TransactionTab(Browser<FinancialAct> browser, TransactionCRUDWindow window, boolean refreshOnShow) {
            super(browser, window, refreshOnShow);
        }

        /**
         * Returns the CRUD window.
         *
         * @return the window
         */
        @Override
        public TransactionCRUDWindow getWindow() {
            return (TransactionCRUDWindow) super.getWindow();
        }

        /**
         * Returns the customer of the selected object.
         *
         * @return the customer, or {@code null} if there is none
         */
        @Override
        public Party getCustomer() {
            return getWindow().getCustomer();
        }

        /**
         * Invoked when an object is double-clicked.
         * <p/>
         * This views the selected object.
         */
        @Override
        protected void onDoubleClick() {
            getWindow().view();
        }
    }

    private static class TableModel extends DescriptorTableModel<FinancialAct> {

        public TableModel(String[] archetypes, LayoutContext context) {
            super(archetypes, context);
        }

        /**
         * Returns a list of descriptor names to include in the table.
         *
         * @return the list of descriptor names to include in the table
         */
        @Override
        protected String[] getNodeNames() {
            return new String[]{"id", "startTime", "customer", "status", "amount", "notes"};
        }
    }
}