/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.patient.info;

import org.openvpms.component.model.party.Party;
import org.openvpms.hl7.patient.PatientContext;
import org.openvpms.hl7.patient.PatientInformationService;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.DefaultIMObjectActions;
import org.openvpms.web.component.im.util.UserHelper;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.component.workflow.DefaultTaskListener;
import org.openvpms.web.component.workflow.TaskEvent;
import org.openvpms.web.component.workflow.Workflow;
import org.openvpms.web.component.workspace.AbstractViewCRUDWindow;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.workflow.WorkflowFactory;
import org.openvpms.web.workspace.workflow.merge.MergeWorkflow;


/**
 * Information CRUD window.
 *
 * @author Tim Anderson
 */
public class InformationCRUDWindow extends AbstractViewCRUDWindow<Party> {

    /**
     * The check-in button identifier.
     */
    private static final String CHECKIN_ID = "checkin";

    /**
     * The merge button identifier.
     */
    private static final String MERGE_ID = "merge";


    /**
     * Constructs an {@link InformationCRUDWindow}.
     *
     * @param archetypes the archetypes that this may create
     * @param context    the context
     * @param help       the help context
     */
    public InformationCRUDWindow(Archetypes<Party> archetypes, Context context, HelpContext help) {
        super(archetypes, DefaultIMObjectActions.getInstance(), context, help);
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        super.layoutButtons(buttons);
        buttons.add("checkin", this::onCheckIn);
        if (UserHelper.isAdmin(getContext().getUser())) {
            // only provide merge for admin users
            buttons.add("merge", this::onMerge);
        }
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        super.enableButtons(buttons, enable);
        buttons.setEnabled(CHECKIN_ID, enable);
        buttons.setEnabled(MERGE_ID, enable);
    }

    /**
     * Invoked when the object has been saved.
     *
     * @param object the object
     * @param isNew  determines if the object is a new instance
     */
    @Override
    protected void onSaved(Party object, boolean isNew) {
        super.onSaved(object, isNew);
        notifyUpdated(object);
    }

    /**
     * Checks in the current patient.
     */
    private void onCheckIn() {
        Context context = getContext();
        Party customer = context.getCustomer();
        Party patient = context.getPatient();
        if (customer != null && patient != null) {
            WorkflowFactory factory = ServiceHelper.getBean(WorkflowFactory.class);
            Workflow workflow = factory.createCheckInWorkflow(customer, patient, context, getHelpContext());
            workflow.start();
        } else {
            String title = Messages.get("patient.checkin.title");
            String msg = Messages.get("patient.checkin.needcustomerpatient");
            ErrorHelper.show(title, msg);
        }
    }

    /**
     * Merges the current patient with another.
     */
    private void onMerge() {
        Party customer = getContext().getCustomer();
        HelpContext help = getHelpContext().subtopic("merge");
        MergeWorkflow workflow = new PatientMergeWorkflow(getObject(), customer, help);
        workflow.addTaskListener(new DefaultTaskListener() {
            /**
             * Invoked when a task event occurs.
             *
             * @param event the event
             */
            public void taskEvent(TaskEvent event) {
                if (event.getType() == TaskEvent.Type.COMPLETED) {
                    onRefresh(getObject());
                    // parent should have updated it
                    Party object = getObject();
                    if (object != null) {
                        notifyUpdated(object);
                    }

                }
            }
        });
        workflow.start();
    }

    /**
     * Notify registered listeners that a patient has updated.
     *
     * @param object the patient
     */
    private void notifyUpdated(Party object) {
        PatientContext context = PatientContextHelper.getPatientContext(object, getContext());
        if (context != null) {
            PatientInformationService service = ServiceHelper.getBean(PatientInformationService.class);
            service.updated(context);
        }
    }
}
