/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.RadioButton;
import nextapp.echo2.app.button.ButtonGroup;
import nextapp.echo2.app.table.TableColumn;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.business.service.archetype.helper.sort.IMObjectSorter;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.paymentprocessor.internal.service.PaymentProcessors;
import org.openvpms.paymentprocessor.service.PaymentProcessorService;
import org.openvpms.web.component.im.delete.Deletable;
import org.openvpms.web.component.im.edit.AbstractRemoveConfirmationHandler;
import org.openvpms.web.component.im.edit.ActCollectionResultSetFactory;
import org.openvpms.web.component.im.edit.CollectionPropertyEditor;
import org.openvpms.web.component.im.edit.IMObjectCollectionEditor;
import org.openvpms.web.component.im.edit.IMObjectEditor;
import org.openvpms.web.component.im.edit.act.ActRelationshipCollectionEditor;
import org.openvpms.web.component.im.edit.payment.PaymentItemEditor;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.table.DescriptorTableColumn;
import org.openvpms.web.component.im.table.IMTableModel;
import org.openvpms.web.component.im.view.TableComponentFactory;
import org.openvpms.web.component.property.CollectionProperty;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.echo.button.ButtonRow;
import org.openvpms.web.echo.dialog.ConfirmationDialog;
import org.openvpms.web.echo.dialog.InformationDialog;
import org.openvpms.web.echo.factory.ButtonFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.charge.DefaultEditorQueue;
import org.openvpms.web.workspace.customer.charge.EditorQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.stream.Collectors;

/**
 * Editor for collections of payment and refund item {@link ActRelationship}s.
 * <p/>
 * This defaults the item archetype to the <em>defaultPaymentType</em> of <em>party.organisationPractice</em>.
 * i.e, if the defaultPayment type is <em>act.customerAccountPaymentEFT</em>, the default refund type is
 * <em>act.customerAccountRefundEFT</em>.
 *
 * @author Tim Anderson
 */
public class PaymentItemRelationshipCollectionEditor extends ActRelationshipCollectionEditor {

    /**
     * The payment processors service.
     */
    private final PaymentProcessors processors;

    /**
     * The editor queue.
     */
    private final EditorQueue queue;

    /**
     * The current payment processor.
     */
    private Entity paymentProcessor;

    /**
     * The parent editor.
     */
    private AbstractCustomerPaymentEditor parent;

    /**
     * The default payment type node of party.organisationPractice.
     */
    private static final String DEFAULT_PAYMENT_TYPE = "defaultPaymentType";

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(PaymentProcessorPaymentItemEditor.class);

    /**
     * Creates a {@link PaymentItemRelationshipCollectionEditor}.
     *
     * @param property the collection property
     * @param act      the parent act
     * @param context  the layout context
     */
    public PaymentItemRelationshipCollectionEditor(CollectionProperty property, Act act, LayoutContext context) {
        super(property, act, context, new CollectionFactory());
        processors = ServiceHelper.getBean(PaymentProcessors.class);
        queue = new DefaultEditorQueue(context.getContext());

        // register a handler to prevent removal of EFT and payment processor items based on their transaction status
        setRemoveConfirmationHandler(new RemoveHandler());
    }

    /**
     * Sets the parent editor.
     * <p/>
     * This is used when processing payment item transactions, in order to change the status from POSTED to IN_PROGRESS.
     *
     * @param parent the parent editor
     */
    public void setParent(AbstractCustomerPaymentEditor parent) {
        this.parent = parent;
    }

    /**
     * Returns the editor queue.
     *
     * @return the queue
     */
    public EditorQueue getQueue() {
        return queue;
    }

    /**
     * Adds a new item to the collection, subject to the constraints of {@link #create(String)}.
     *
     * @param shortName the archetype to add
     * @return the editor for the item, or {@code null} a new item cannot be created.
     */
    @Override
    public IMObjectEditor add(String shortName) {
        IMObjectEditor editor = super.add(shortName);
        if (editor != null) {
            editor.getComponent();
            // clear any modified flags so the object can be replaced
            editor.clearModified();
        }
        return editor;
    }

    /**
     * Creates a new editor.
     *
     * @param object  the object to edit
     * @param context the layout context
     * @return an editor to edit {@code object}
     */
    @Override
    public IMObjectEditor createEditor(IMObject object, LayoutContext context) {
        IMObjectEditor editor = super.createEditor(object, context);
        initialise(editor);
        return editor;
    }

    /**
     * Returns the payment item editors.
     *
     * @return the payment item editors
     */
    public List<PaymentItemEditor> getPaymentItemEditors() {
        return getCurrentActs().stream()
                .map(this::getEditor)
                .filter(editor -> editor instanceof PaymentItemEditor)
                .map(editor -> (PaymentItemEditor) editor)
                .collect(Collectors.toList());
    }

    /**
     * Initialises a new editor.
     *
     * @param editor the editor
     */
    protected void initialise(IMObjectEditor editor) {
        if (editor instanceof PaymentProcessorPaymentItemEditor) {
            PaymentProcessorPaymentItemEditor paymentItemEditor = (PaymentProcessorPaymentItemEditor) editor;
            if (paymentItemEditor.getPaymentProcessor() == null) {
                paymentItemEditor.setPaymentProcessor(paymentProcessor);
            }
            paymentItemEditor.setPaymentEditor(parent);
        }
    }

    /**
     * Validates the object.
     * <p>
     * This validates the current object being edited, and if valid, the collection.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        return super.doValidation(validator) && queue.isComplete()
               && validateSplitTransactions(validator);
    }

    /**
     * Ensures that if an editor doesn't support split transactions and multiple items are present, a validation error
     * is raised.
     *
     * @param validator the validator
     * @return {@code true} if valid, otherwise {@code false}
     */
    protected boolean validateSplitTransactions(Validator validator) {
        List<PaymentItemEditor> paymentItemEditors = getPaymentItemEditors();
        if (paymentItemEditors.size() > 1) {
            paymentItemEditors.stream()
                    .filter(editor -> !editor.supportsSplitTransactions())
                    .findFirst()
                    .ifPresent(editor -> {
                        boolean isPayment = getObject().isA(CustomerAccountArchetypes.PAYMENT);
                        String key = isPayment ? "customer.payment.splitPaymentUnsupported"
                                               : "customer.payment.splitRefundUnsupported";
                        validator.add(this, Messages.format(key, editor.getDisplayName()));
                    });
        }
        return validator.isValid();
    }

    /**
     * Lays out the component.
     *
     * @param context the layout context
     * @return the component
     */
    @Override
    protected Component doLayout(LayoutContext context) {
        Component component = super.doLayout(context);
        String archetype = getDefaultArchetype();
        if (archetype != null) {
            setArchetype(archetype);
        }
        return component;
    }

    /**
     * Determines if the previous and next navigation buttons should be displayed.
     *
     * @return {@code false}
     */
    @Override
    protected boolean showPreviousNext() {
        return false;
    }

    /**
     * Enable/disables the buttons.
     *
     * @param enable if {@code true} enable buttons (subject to criteria), otherwise disable them
     */
    @Override
    protected void enableNavigation(boolean enable) {
        boolean enableAdd = enable;
        if (enable) {
            // only enable Add if all existing editors support split transactions
            enableAdd = getPaymentItemEditors().stream().allMatch(PaymentItemEditor::supportsSplitTransactions);
        }
        super.enableNavigation(enable, enableAdd);
    }

    /***
     * Adds an archetype selector to the button row, if there is more than one archetype.
     *
     * @param buttons the buttons
     * @param focus   the focus group
     */
    @Override
    protected void addArchetypeSelector(ButtonRow buttons, FocusGroup focus) {
        ArchetypeService service = getService();
        String[] range = getArchetypes();
        ButtonGroup group = new ButtonGroup();
        String defaultArchetype = getDefaultArchetype();
        if (defaultArchetype == null && range.length > 0) {
            defaultArchetype = range[0];
        }
        for (String archetype : range) {
            if (TypeHelper.isA(archetype, CustomerAccountArchetypes.PAYMENT_PP, CustomerAccountArchetypes.REFUND_PP)) {
                addPaymentProcessors(archetype, buttons, group);
            } else {
                String displayName = DescriptorHelper.getDisplayName(archetype, service);
                RadioButton button = ButtonFactory.text(displayName, group, () -> onArchetypeSelected(archetype, null));
                if (archetype.equals(defaultArchetype)) {
                    setArchetype(archetype);
                    button.setSelected(true);
                }
                buttons.addButton(button);
            }
        }
    }

    /**
     * Returns the archetypes this collection supports.
     *
     * @return the archetypes
     */
    protected String[] getArchetypes() {
        return DescriptorHelper.getShortNames(getCollectionPropertyEditor().getArchetypeRange(), false, getService());
    }

    /**
     * Invoked when on archetype is selected.
     * <p/>
     * If the current object is new and unmodified, it will be replaced by an object of the specified archetype.
     * If not, it specifies the archetype that will be created when clicking Add.
     *
     * @param archetype        the archetype
     * @param paymentProcessor the payment processor, or {@code null} if the archetype doesn't support payment
     *                         processors
     * @return the new editor, or {@code null} if one could not be created
     */
    protected IMObjectEditor onArchetypeSelected(String archetype, Entity paymentProcessor) {
        IMObjectEditor result = null;
        setArchetype(archetype);
        this.paymentProcessor = paymentProcessor;
        IMObjectEditor currentEditor = getCurrentEditor();
        IMObject object = (currentEditor != null) ? currentEditor.getObject() : null;
        if (object != null && object.isNew() && !currentEditor.isModified()) {
            // existing object hasn't been changed from its default values, so replace it
            remove(object);
        }
        if (checkSplitTransactions(paymentProcessor)) {
            result = onAdd();
        }
        return result;
    }

    /**
     * Verifies that a payment item can be added.
     *
     * @param paymentProcessor the payment processor, if a payment processor item is being added, otherwise {@code null}
     * @return if the item can be added, otherwise {@code false}
     */
    private boolean checkSplitTransactions(Entity paymentProcessor) {
        boolean valid = true;
        List<PaymentItemEditor> editors = getPaymentItemEditors();
        if (paymentProcessor != null && !editors.isEmpty()) {
            splitTransactionUnsupported(paymentProcessor.getName());
            valid = false;
        } else {
            for (PaymentItemEditor editor : editors) {
                if (!editor.supportsSplitTransactions()) {
                    splitTransactionUnsupported(editor.getDisplayName());
                    valid = false;
                    break;
                }
            }
        }
        return valid;
    }

    /**
     * Displays a dialog indicating that split transactions are not supported
     *
     * @param displayName the display name for the editor that doesn't support split transactions
     */
    private void splitTransactionUnsupported(String displayName) {
        boolean isPayment = getObject().isA(CustomerAccountArchetypes.PAYMENT);
        String key = isPayment ? "customer.payment.splitPaymentUnsupported"
                               : "customer.payment.splitRefundUnsupported";
        InformationDialog.show(Messages.format(key, displayName));
    }

    /**
     * Creates radio buttons to add new payment items for payment processors.
     * <p/>
     * The payment processor name is used as the button text.
     * <p/>
     * For refunds, buttons are only added if the payment processor supports them
     *
     * @param archetype the <em>act.customerAccountPaymentPP</em> or <em>act.customerAccountRefundPP</em> archetype
     * @param buttons   the buttons to add to
     * @param group     the button group
     */
    private void addPaymentProcessors(String archetype, ButtonRow buttons, ButtonGroup group) {
        boolean refund = TypeHelper.isA(archetype, CustomerAccountArchetypes.REFUND_PP);
        List<Entity> configs = ServiceHelper.getBean(PaymentProcessors.class).getPaymentProcessors();
        if (!configs.isEmpty()) {
            configs.sort(IMObjectSorter.getNameComparator().thenComparing(IMObjectSorter.getIdComparator()));
            for (Entity config : configs) {
                if (!refund || paymentProcessorSupportsRefunds(config)) {
                    RadioButton button = ButtonFactory.text(config.getName(), group,
                                                            () -> onArchetypeSelected(archetype, config));
                    buttons.addButton(button);
                }
            }
        }
    }

    /**
     * Determines if a payment processor supports refunds.
     *
     * @param config the payment processor configuration
     * @return {@code true} if the payment processor supports refunds, otherwise {@code false}
     */
    private boolean paymentProcessorSupportsRefunds(Entity config) {
        boolean result = false;
        try {
            PaymentProcessorService processor = processors.getPaymentProcessor(config);
            result = processor.getRefundCapabilities().isSupported();
        } catch (Throwable exception) {
            log.debug("Failed to retrieve PaymentProcessorService for {} ({}): {}", config.getName(), config.getId(),
                      exception.getMessage(), exception);
        }
        return result;
    }

    /**
     * Returns the default item archetype.
     *
     * @return the default item archetype
     */
    private String getDefaultArchetype() {
        String archetype = null;
        PracticeService practiceService = ServiceHelper.getBean(PracticeService.class);
        Party practice = practiceService.getPractice();
        if (practice != null) {
            IMObjectBean bean = getBean(practice);
            archetype = bean.getString(DEFAULT_PAYMENT_TYPE);
            if (archetype != null && getObject().isA(CustomerAccountArchetypes.REFUND)) {
                archetype = ServiceHelper.getBean(CustomerAccountRules.class).getReversalArchetype(archetype);
            }
        }
        return archetype;
    }

    private static class RemoveHandler extends AbstractRemoveConfirmationHandler {

        /**
         * Displays a confirmation dialog to confirm removal of an object from a collection.
         * <p>
         * If approved, it performs the removal.
         *
         * @param object     the object to remove
         * @param collection the collection to remove the object from, if approved
         */
        @Override
        protected void confirmRemove(IMObject object, IMObjectCollectionEditor collection) {
            PaymentItemRelationshipCollectionEditor editor = (PaymentItemRelationshipCollectionEditor) collection;
            IMObjectEditor itemEditor = editor.getEditor(object);
            if (itemEditor instanceof TransactionPaymentItemEditor) {
                TransactionPaymentItemEditor transactionPaymentItemEditor = (TransactionPaymentItemEditor) itemEditor;
                Deletable deletable = transactionPaymentItemEditor.getDeletable();
                if (deletable.canDelete()) {
                    super.confirmRemove(object, collection);
                } else {
                    String displayName = transactionPaymentItemEditor.getTransactionDisplayName();
                    if (transactionPaymentItemEditor.canCancelTransaction()) {
                        ConfirmationDialog.newDialog()
                                .titleKey("customer.payment.delete.cancel.title", displayName)
                                .messageKey("customer.payment.delete.cancel.message", displayName)
                                .yesNo()
                                .yes(() -> {
                                    transactionPaymentItemEditor.cancelTransaction();
                                    Deletable postCancel = transactionPaymentItemEditor.getDeletable();
                                    if (postCancel.canDelete()) {
                                        apply(object, collection);
                                    } else {
                                        cannotDelete(postCancel, displayName);
                                        cancelRemove(collection);
                                    }
                                })
                                .no(() -> cancelRemove(collection))
                                .show();
                    } else {
                        cannotDelete(deletable, displayName);
                        cancelRemove(collection);
                    }
                }
            } else {
                super.confirmRemove(object, collection);
            }
        }

        /**
         * Invoked when an object cannot be deleted.
         *
         * @param status      the deletable status
         * @param displayName the display name
         */
        private void cannotDelete(Deletable status, String displayName) {
            String title = Messages.format("imobject.collection.delete.title", displayName);
            InformationDialog.show(title, status.getReason());
        }
    }

    private static class CollectionFactory extends ActCollectionResultSetFactory {
        /**
         * Creates a table model to display the result set.
         * <p/>
         * This implementation sets the default sort column to the first column, and sorts it descending if
         * it is a "startTime" node.
         *
         * @param property the collection property
         * @param parent   the parent object
         * @param context  the layout context
         * @return a new table model
         */
        @Override
        @SuppressWarnings("unchecked")
        public IMTableModel<IMObject> createTableModel(CollectionPropertyEditor property, IMObject parent,
                                                       LayoutContext context) {
            context = new DefaultLayoutContext(context);
            context.setComponentFactory(new TableComponentFactory(context));
            PaymentItemTableModel model = new PaymentItemTableModel(property.getArchetypeRange(), context);
            TableColumn column = model.getColumnModel().getColumn(0);
            model.setDefaultSortAscending(!(column instanceof DescriptorTableColumn) || !((DescriptorTableColumn) column).getName().equals("startTime"));
            model.setDefaultSortColumn(column.getModelIndex());
            return (IMTableModel<IMObject>) (IMTableModel<?>) model;
        }
    }
}
