/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.communication;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.table.DescriptorTableColumn;
import org.openvpms.web.component.im.table.act.AbstractActTableModel;
import org.openvpms.web.resource.i18n.format.DateFormatter;

import java.util.Date;

import static org.openvpms.web.workspace.customer.communication.CommunicationLayoutStrategy.ADDRESS;
import static org.openvpms.web.workspace.customer.communication.CommunicationLayoutStrategy.START_TIME;


/**
 * Table model for <em>act.customerCommunication*</em> acts.
 *
 * @author Tim Anderson
 */
public class CommunicationTableModel extends AbstractActTableModel<Act> {

    /**
     * The node descriptor names to display in the table.
     */
    private static final String[] NODES = {START_TIME,
                                           CommunicationLayoutStrategy.DESCRIPTION,
                                           CommunicationLayoutStrategy.CREATED_BY,
                                           ADDRESS,
                                           CommunicationLayoutStrategy.REASON,
                                           CommunicationLayoutStrategy.LOCATION};


    /**
     * Constructs a {@link CommunicationTableModel}.
     *
     * @param shortNames the act archetype short names
     * @param context    the layout context. May be {@code null}
     */
    public CommunicationTableModel(String[] shortNames, LayoutContext context) {
        super(shortNames, context);
    }

    /**
     * Returns a list of node descriptor names to include in the table.
     *
     * @return the list of node descriptor names to include in the table
     */
    @Override
    protected String[] getNodeNames() {
        return NODES;
    }

    /**
     * Returns a value for a given column.
     *
     * @param object the object to operate on
     * @param column the column
     * @param row    the row
     * @return the value for the column
     */
    @Override
    protected Object getValue(Act object, DescriptorTableColumn column, int row) {
        Object result;
        if (START_TIME.equals(column.getName())) {
            result = getCreatedTime(object);
        } else if (ADDRESS.equals(column.getName())) {
            result = getAddress(object);
        } else {
            result = super.getValue(object, column, row);
        }
        return result;
    }

    /**
     * Returns the first line of the address, if present.
     *
     * @param act the communication act
     * @return the address. May be {@code null}
     */
    private String getAddress(Act act) {
        IMObjectBean bean = getBean(act);
        String result = null;
        if (bean.hasNode(ADDRESS)) {
            result = bean.getString(ADDRESS);
            if (!StringUtils.isEmpty(result)) {
                String abbr = StringUtils.substringBefore(result, "\n");
                if (result.length() != abbr.length()) {
                    result = abbr + "...";
                }
            }
        } else if (bean.hasNode("phone")) {
            result = bean.getString("phone");
        }
        return result;
    }

    /**
     * Formats the creation time of the act.
     *
     * @param act the communication act
     * @return the formatted creation time
     */
    private String getCreatedTime(Act act) {
        Date date = act.getCreated();
        return date != null ? DateFormatter.formatDateTimeAbbrev(date) : null;
    }

}
