/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.charge;

import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.patient.prescription.PrescriptionRules;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.web.component.edit.AlertListener;
import org.openvpms.web.component.im.edit.ActCollectionResultSetFactory;
import org.openvpms.web.component.im.edit.CollectionResultSetFactory;
import org.openvpms.web.component.im.edit.IMObjectEditor;
import org.openvpms.web.component.im.edit.act.ActItemEditor;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.product.PricingContext;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.property.CollectionProperty;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.echo.dialog.ConfirmationDialog;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.PriceActItemEditor;
import org.openvpms.web.workspace.customer.StockOnHand;
import org.openvpms.web.workspace.customer.charge.department.DepartmentListener;
import org.openvpms.web.workspace.patient.charge.TemplateChargeItems;
import org.openvpms.web.workspace.patient.mr.Prescriptions;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;


/**
 * Editor for <em>actRelationship.customerAccountInvoiceItem</em> and
 * <em>actRelationship.customerAccountCreditItem</em> act relationships.
 * Sets a {@link EditorQueue} on {@link CustomerChargeActItemEditor} instances.
 *
 * @author Tim Anderson
 */
public class ChargeItemRelationshipCollectionEditor extends AbstractChargeItemRelationshipCollectionEditor {

    /**
     * The templates.
     */
    private final List<TemplateChargeItems> templates = new ArrayList<>();

    /**
     * Listener for department changes.
     */
    private final DepartmentListener departmentListener;

    /**
     * Last Selected Item Date.
     */
    private Date lastItemDate = null;

    /**
     * Listener invoked when {@link #onAdd()} is invoked.
     */
    private Runnable listener;

    /**
     * The alert identifier, used to cancel any existing alert.
     */
    private String alertId;

    /**
     * The start time node name.
     */
    private static final String START_TIME = "startTime";

    /**
     * Constructs a {@link ChargeItemRelationshipCollectionEditor}.
     *
     * @param property the collection property
     * @param act      the parent act
     * @param context  the layout context
     */
    public ChargeItemRelationshipCollectionEditor(CollectionProperty property, Act act, LayoutContext context) {
        this(property, act, context, ActCollectionResultSetFactory.INSTANCE);
    }

    /**
     * Constructs a {@link ChargeItemRelationshipCollectionEditor}.
     *
     * @param property the collection property
     * @param act      the parent act
     * @param context  the layout context
     */
    public ChargeItemRelationshipCollectionEditor(CollectionProperty property, Act act, LayoutContext context,
                                                  CollectionResultSetFactory factory) {
        super(property, act, context, factory, createEditContext(act, context));
        Prescriptions prescriptions = null;
        CustomerChargeEditContext editContext = getEditContext();
        if (act.isA(CustomerAccountArchetypes.INVOICE)) {
            List<Act> items = getCurrentActs();
            InvestigationManager investigationManager = editContext.getInvestigations();
            for (Act item : items) {
                investigationManager.addInvoiceItem((FinancialAct) item);
            }
            prescriptions = new Prescriptions(items, ServiceHelper.getBean(PrescriptionRules.class),
                                              ServiceHelper.getArchetypeService());
            setRemoveConfirmationHandler(new InvoiceRemoveConfirmationHandler(
                    investigationManager, context.getContext(), context.getHelpContext()));
        } else {
            setRemoveConfirmationHandler(new DefaultChargeRemoveConfirmationHandler(context.getContext(),
                                                                                    context.getHelpContext()));
        }
        editContext.setPrescriptions(prescriptions);
        departmentListener = editContext.useDepartments() ? this::onDepartmentChanged : null;
    }

    /**
     * Returns the save context.
     *
     * @return the save context
     */
    public ChargeSaveContext getSaveContext() {
        return getEditContext().getSaveContext();
    }

    /**
     * Sets the popup editor manager.
     *
     * @param queue the popup editor manager. May be {@code null}
     */
    public void setEditorQueue(EditorQueue queue) {
        getEditContext().setEditorQueue(queue);
    }

    /**
     * Returns the popup editor manager.
     *
     * @return the popup editor manager. May be {@code null}
     */
    public EditorQueue getEditorQueue() {
        return getEditContext().getEditorQueue();
    }

    /**
     * Removes an object from the collection.
     *
     * @param object the object to remove
     */
    @Override
    public void remove(IMObject object) {
        super.remove(object);
        FinancialAct act = (FinancialAct) object;
        CustomerChargeEditContext editContext = getEditContext();
        editContext.getStock().remove(act);
        Prescriptions prescriptions = editContext.getPrescriptions();
        if (prescriptions != null) {
            prescriptions.removeItem(act);
        }
        InvestigationManager investigations = getEditContext().getInvestigations();
        investigations.removeInvoiceItem(act);
        Alerts alerts = editContext.getAlerts();
        alerts.removeItem(act);

        EditorQueue queue = getEditorQueue();
        if (queue != null) {
            // cancel any editors, popups and callbacks related to the object
            queue.cancel(object);
        }
    }

    /**
     * Registers a listener that is invoked when the user adds an item.
     * <p/>
     * Note that this is not invoked for template expansion.
     *
     * @param listener the listener to invoke. May be {@code null}
     */
    public void setAddItemListener(Runnable listener) {
        this.listener = listener;
    }

    /**
     * Returns the templates that were expanded.
     *
     * @return the templates
     */
    public List<TemplateChargeItems> getTemplates() {
        return templates;
    }

    /**
     * Clears the templates.
     */
    public void clearTemplates() {
        templates.clear();
    }

    /**
     * Creates a new editor.
     *
     * @param object  the object to edit
     * @param context the layout context
     * @return an editor to edit {@code object}
     */
    @Override
    public IMObjectEditor createEditor(IMObject object, LayoutContext context) {
        return initialiseEditor(new DefaultCustomerChargeActItemEditor((FinancialAct) object, (Act) getObject(),
                                                                       getEditContext(), context));
    }

    /**
     * Returns an editor for an object, creating one if it doesn't exist.
     *
     * @param object the object to edit
     * @return an editor for the object
     */
    @Override
    public CustomerChargeActItemEditor getEditor(IMObject object) {
        return (CustomerChargeActItemEditor) super.getEditor(object);
    }

    /**
     * Invoked when the parent charge is POSTED.
     * <p/>
     * This updates the endTime (i.e. the Completed Date) on each of the items.
     */
    public void posted(Date date) {
        for (Act act : getCurrentActs()) {
            act.setActivityEndTime(date);
        }
    }

    /**
     * Returns the pricing context.
     *
     * @return the pricing context
     */
    public PricingContext getPricingContext() {
        return getEditContext().getPricingContext();
    }

    /**
     * Determines if the object has been modified.
     *
     * @return {@code true} if the object has been modified
     */
    @Override
    public boolean isModified() {
        return super.isModified() || getEditContext().getInvestigations().isModified();
    }

    /**
     * Validates the object.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        if (super.doValidation(validator) && validateInvestigations(validator)) {
            EditorQueue queue = getEditorQueue();
            return (queue == null || queue.isComplete());
        }
        return false;
    }

    /**
     * Invoked when the "Add" button is pressed. Creates a new instance of the selected archetype, and displays it in
     * an editor.
     *
     * @return the new editor, or {@code null} if one could not be created
     */
    @Override
    protected IMObjectEditor onAdd() {
        unmarkAll();
        IMObjectEditor editor = add();
        if (editor != null && listener != null) {
            EditorQueue queue = getEditorQueue();
            if (!queue.isComplete()) {
                queue.queue(() -> {
                    if (listener != null) {
                        listener.run();
                    }
                });
            } else {
                listener.run();
            }
        }
        return editor;
    }

    /**
     * Returns the edit context.
     *
     * @return the edit context
     */
    @Override
    protected CustomerChargeEditContext getEditContext() {
        return (CustomerChargeEditContext) super.getEditContext();
    }

    /**
     * Initialises an editor.
     *
     * @param editor the editor
     */
    protected CustomerChargeActItemEditor initialiseEditor(CustomerChargeActItemEditor editor) {
        // set startTime to the last used value
        if (lastItemDate != null) {
            editor.getProperty(START_TIME).setValue(lastItemDate);
        }

        // add a listener to store the last used item startTime.
        ModifiableListener startTimeListener = modifiable -> lastItemDate = editor.getProperty(START_TIME).getDate();
        editor.getProperty(START_TIME).addModifiableListener(startTimeListener);
        editor.setProductListener(getProductListener());        // register the listener to expand templates
        editor.setDepartmentListener(departmentListener);
        editor.setAlertListener(getAlertListener());
        return editor;
    }

    /**
     * Edit an object.
     *
     * @param object the object to edit
     * @return the editor
     */
    @Override
    protected IMObjectEditor edit(IMObject object) {
        CustomerChargeActItemEditor editor = (CustomerChargeActItemEditor) super.edit(object);
        editor.updateOnHandQuantity();
        return editor;
    }

    /**
     * Copies an act item for each product referred to in its template.
     *
     * @param editor   the editor
     * @param template the product template
     * @param quantity the quantity
     * @return the acts generated from the template
     */
    @Override
    protected List<Act> createTemplateActs(ActItemEditor editor, Product template, BigDecimal quantity) {
        List<Act> acts = super.createTemplateActs(editor, template, quantity);
        if (!acts.isEmpty()) {
            TemplateChargeItems items = new TemplateChargeItems(template, acts);
            templates.add(items);
        }

        AlertListener alertListener = getAlertListener();
        if (alertListener != null && !acts.isEmpty()) {
            int outOfStock = 0;
            StockOnHand stock = getEditContext().getStock();
            for (Act act : acts) {
                BigDecimal onHand = stock.getAvailableStock((FinancialAct) act);
                if (onHand != null && BigDecimal.ZERO.compareTo(onHand) >= 0) {
                    ++outOfStock;
                }
            }
            if (outOfStock != 0) {
                if (alertId != null) {
                    alertListener.cancel(alertId);
                    alertId = null;
                }
                alertId = alertListener.onAlert(Messages.format("customer.charge.outofstock", outOfStock));
            }
        }
        EditorQueue queue = getEditorQueue();
        if (!acts.isEmpty() && !queue.isComplete()) {
            // the collection is considered invalid while there are popups, so force a validation check when
            // popups have all closed, and move focus to the product
            queue.queue(() -> {
                // need to notify listeners as the child editors don't notify the parent during queueing
                getListeners().notifyListeners(this);
                isValid();
                IMObjectEditor currentEditor = getCurrentEditor();
                if (currentEditor instanceof PriceActItemEditor) {
                    ((PriceActItemEditor) currentEditor).moveFocusToProduct();
                }
            });
        }
        return acts;
    }

    /**
     * Saves any current edits.
     *
     * @throws OpenVPMSException if the save fails
     */
    @Override
    protected void doSave() {
        CustomerChargeEditContext editContext = getEditContext();

        // Need to save prescriptions and investigations first, as invoice item deletion can cause
        // StaleObjectStateExceptions otherwise
        Prescriptions prescriptions = editContext.getPrescriptions();
        if (prescriptions != null) {
            prescriptions.save();
        }
        InvestigationManager investigations = editContext.getInvestigations();
        investigations.save();

        super.doSave();

        // clear the stock on hand so it is recreated from saved state
        editContext.getStock().clear();
    }

    /**
     * Creates a new {@link ChargeItemTableModel}.
     *
     * @param context the layout context
     * @return a new table model
     */
    @Override
    protected ChargeItemTableModel<Act> createChargeItemTableModel(LayoutContext context) {
        CustomerChargeEditContext editContext = getEditContext();
        return new ChargeItemTableModel<>(getCollectionPropertyEditor().getArchetypeRange(), editContext.getStock(),
                                          context);
    }

    /**
     * Invoked when a department changes.
     *
     * @param editor     the editor
     * @param department the department. May be {@code null}
     */
    private void onDepartmentChanged(CustomerChargeActItemEditor editor, Entity department) {
        List<IMObject> marked = getMarked();
        marked.remove(editor.getObject());
        if (marked.isEmpty()) {
            Entity template = editor.getTemplate();
            if (template != null) {
                List<CustomerChargeActItemEditor> editors = getEditorsForTemplate(editor, template, department);
                if (!editors.isEmpty()) {
                    ConfirmationDialog dialog = ConfirmationDialog.newDialog()
                            .title(Messages.get("customer.charge.department.change.title"))
                            .message(Messages.format("customer.charge.department.change.template", editors.size(),
                                                     template.getName()))
                            .yesNo()
                            .yes(() -> changeDepartments(editors, department))
                            .build();
                    getEditorQueue().queue(dialog);
                }
            }
        } else {
            List<CustomerChargeActItemEditor> editors = new ArrayList<>();
            for (IMObject object : marked) {
                editors.add(getEditor(object));
            }
            ConfirmationDialog dialog = ConfirmationDialog.newDialog()
                    .title(Messages.get("customer.charge.department.change.title"))
                    .message(Messages.format("customer.charge.department.change.message", editors.size()))
                    .yesNo()
                    .yes(() -> changeDepartments(editors, department))
                    .build();
            getEditorQueue().queue(dialog);
        }
    }

    /**
     * Returns other editors generated by a template that have a different department to that specified.
     *
     * @param editor     the editor to exclude
     * @param template   the template
     * @param department the department. May be {@code null}
     * @return other editors generated by the template with a different department
     */
    private List<CustomerChargeActItemEditor> getEditorsForTemplate(CustomerChargeActItemEditor editor, Entity template,
                                                                    Entity department) {
        Reference templateRef = template.getObjectReference();
        int group = editor.getTemplateGroup();
        List<CustomerChargeActItemEditor> editors = new ArrayList<>();
        if (group != -1) {
            Reference departmentRef = (department != null) ? department.getObjectReference() : null;
            for (Act object : getCurrentActs()) {
                if (object != editor.getObject()) {
                    CustomerChargeActItemEditor other = getEditor(object);
                    if (Objects.equals(templateRef, other.getTemplateRef()) && group == other.getTemplateGroup()
                        && !Objects.equals(other.getDepartmentRef(), departmentRef)) {
                        editors.add(other);
                    }
                }
            }
        }
        return editors;
    }

    /**
     * Change departments on the supplied editors.
     *
     * @param editors    the editors to update
     * @param department the new department
     */
    private void changeDepartments(List<CustomerChargeActItemEditor> editors, Entity department) {
        for (CustomerChargeActItemEditor editor : editors) {
            editor.setDepartmentListener(null);
            try {
                editor.setDepartment(department);
            } finally {
                editor.setDepartmentListener(departmentListener);
            }
        }
        unmarkAll();
    }

    /**
     * Validates investigations.
     *
     * @param validator the validator
     * @return {@code true} if the investigations are valid otherwise {@code false}
     */
    private boolean validateInvestigations(Validator validator) {
        InvestigationManager investigationManager = getEditContext().getInvestigations();
        return investigationManager.validate(validator);
    }

    /**
     * Creates a new charge edit context.
     *
     * @param act     the parent charge
     * @param context the layout context
     * @return a new charge edit context
     */
    private static CustomerChargeEditContext createEditContext(Act act, LayoutContext context) {
        IMObjectBean bean = ServiceHelper.getArchetypeService().getBean(act);
        Party customer = (Party) IMObjectHelper.getObject(bean.getTargetRef("customer"), context.getContext());
        if (customer == null) {
            throw new IllegalStateException(act.getArchetype() + " has no customer");
        }
        Party location = (Party) IMObjectHelper.getObject(bean.getTargetRef("location"), context.getContext());
        return new CustomerChargeEditContext(customer, location, context);
    }

}
