/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.supplier.order;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.Grid;
import nextapp.echo2.app.SelectField;
import org.openvpms.archetype.rules.stock.StockArchetypes;
import org.openvpms.archetype.rules.supplier.SupplierArchetypes;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.AbstractArchetypeQuery;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.list.IMObjectListCellRenderer;
import org.openvpms.web.component.im.list.IMObjectListModel;
import org.openvpms.web.component.im.query.QueryHelper;
import org.openvpms.web.echo.factory.ColumnFactory;
import org.openvpms.web.echo.factory.GridFactory;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.factory.SelectFieldFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.style.Styles;

import java.util.List;

/**
 * Stock Location and supplier selector.
 *
 * @author Tim Anderson
 */
public class StockLocationSupplierSelector {

    /**
     * The available stock locations.
     */
    private final List<IMObject> stockLocations;

    /**
     * The available suppliers.
     */
    private final List<IMObject> suppliers;

    /**
     * The stock locations.
     */
    private final SelectField stockLocation;

    /**
     * The suppliers.
     */
    private final SelectField supplier;

    /**
     * The component.
     */
    private final Component component;

    /**
     * The focus group.
     */
    private final FocusGroup group;


    /**
     * Creates a new {@code StockLocationSupplierSelector}.
     *
     * @param context the context
     */
    public StockLocationSupplierSelector(Context context) {
        stockLocations = query(StockArchetypes.STOCK_LOCATION);
        stockLocation = SelectFieldFactory.create(new IMObjectListModel(stockLocations, true, false),
                                                  context.getStockLocation());
        stockLocation.setCellRenderer(IMObjectListCellRenderer.NAME);

        suppliers = query(SupplierArchetypes.SUPPLIER_ORGANISATION, SupplierArchetypes.SUPPLIER_PERSON);
        supplier = SelectFieldFactory.create(new IMObjectListModel(suppliers, true, false));
        supplier.setCellRenderer(IMObjectListCellRenderer.NAME);

        Grid grid = GridFactory.create(2);
        grid.add(LabelFactory.create("product.stockLocation"));
        grid.add(stockLocation);
        grid.add(LabelFactory.create("supplier.type"));
        grid.add(supplier);
        component = ColumnFactory.create(Styles.INSET, grid);
        group = new FocusGroup("StockLocationSupplierSelector");
        group.add(0, stockLocation);
        group.add(1, supplier);
        group.setFocus();
    }

    /**
     * Returns the component.
     *
     * @return the component
     */
    public Component getComponent() {
        return component;
    }

    /**
     * Returns the available stock locations.
     *
     * @return the available stock locations
     */
    public List<IMObject> getStockLocations() {
        return stockLocations;
    }

    /**
     * Returns the available suppliers.
     *
     * @return the available suppliers
     */
    public List<IMObject> getSuppliers() {
        return suppliers;
    }

    /**
     * Returns the stock location.
     *
     * @return the stock location, or {@code null} if none is selected
     */
    public Party getStockLocation() {
        return (Party) stockLocation.getSelectedItem();
    }

    /**
     * Returns the supplier.
     *
     * @return the supplier, or {@code null} if none is selected
     */
    public Party getSupplier() {
        return (Party) supplier.getSelectedItem();
    }

    /**
     * Returns the focus group.
     *
     * @return the focus group
     */
    public FocusGroup getFocusGroup() {
        return group;
    }

    /**
     * Queries the specified archetypes.
     *
     * @param archetypes the archetypes
     * @return the objects matching the query
     */
    private List<IMObject> query(String... archetypes) {
        ArchetypeQuery query = new ArchetypeQuery(archetypes, false, true)
                .add(Constraints.sort("name"))
                .add(Constraints.sort("id"))
                .setMaxResults(AbstractArchetypeQuery.ALL_RESULTS);
        return QueryHelper.query(query);
    }

}
