/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.supplier.order;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.component.processor.BatchProcessorListener;
import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.supplier.OrderRules;
import org.openvpms.archetype.rules.supplier.OrderStatus;
import org.openvpms.archetype.rules.supplier.SupplierRules;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.esci.adapter.client.OrderServiceAdapter;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.SaveHelper;
import org.openvpms.web.component.processor.BatchProcessorDialog;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.component.workspace.ActPoster;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.dialog.InformationDialog;
import org.openvpms.web.echo.dialog.PopupDialogListener;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.supplier.SelectStockDetailsDialog;
import org.openvpms.web.workspace.supplier.SupplierHelper;

import java.util.Collections;
import java.util.List;


/**
 * CRUD window for supplier orders.
 *
 * @author Tim Anderson
 */
public class OrderCRUDWindow extends ESCISupplierCRUDWindow {

    /**
     * Copy button identifier.
     */
    private static final String COPY_ID = "copy";

    /**
     * Generate orders button identifier.
     */
    private static final String GENERATE_ID = "button.generateOrders";

    /**
     * Constructs a {@link OrderCRUDWindow}.
     *
     * @param archetypes the archetypes that this may create
     * @param context    the context
     * @param help       the help context
     */
    public OrderCRUDWindow(Archetypes<FinancialAct> archetypes, Context context, HelpContext help) {
        super(archetypes, OrderActions.INSTANCE, context, help);
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        super.layoutButtons(buttons);
        buttons.add(createPostButton());
        buttons.add(createPrintButton());
        buttons.add(createMailButton());
        buttons.add(COPY_ID, this::onCopy);
        buttons.add(GENERATE_ID, this::onGenerate);
        buttons.add(createCheckInboxButton());
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        boolean editEnabled = false;
        boolean deletePostEnabled = false;
        if (enable) {
            FinancialAct object = getObject();
            editEnabled = getActions().canEdit(object);
            String status = object.getStatus();
            deletePostEnabled = !OrderStatus.POSTED.equals(status) && !OrderStatus.ACCEPTED.equals(status)
                                && !OrderStatus.CANCELLED.equals(status);
        }
        buttons.setEnabled(EDIT_ID, editEnabled);
        buttons.setEnabled(DELETE_ID, deletePostEnabled);
        buttons.setEnabled(POST_ID, deletePostEnabled);
        enablePrintPreview(buttons, enable);
        buttons.setEnabled(COPY_ID, enable);
    }

    /**
     * Invoked when a new order has been created.
     * <p/>
     * This implementation pops up a dialog to select the supplier and stock location, then displays an edit dialog for
     * the act.
     *
     * @param act the new act
     */
    @Override
    protected void onCreated(final FinancialAct act) {
        String title = Messages.format("supplier.order.selectdetails.title", getDisplayName(act));
        SelectStockDetailsDialog dialog = new SelectStockDetailsDialog(title, getContext(),
                                                                       getHelpContext().subtopic("new"));
        dialog.addWindowPaneListener(new PopupDialogListener() {
            @Override
            public void onOK() {
                Party supplier = dialog.getSupplier();
                Party location = dialog.getStockLocation();
                addParticipations(act, supplier, location);
                edit(act, null);
            }
        });
        dialog.show();
    }

    /**
     * Invoked when the 'copy' button is pressed.
     */
    protected void onCopy() {
        try {
            OrderRules rules = SupplierHelper.createOrderRules(getContext().getPractice());
            FinancialAct object = getObject();
            String title = null;
            if (!StringUtils.isEmpty(object.getTitle())) {
                title = Messages.format("supplier.order.copy.notes", object.getTitle());
            }
            FinancialAct copy = rules.copyOrder(object, title);
            edit(copy, null);
        } catch (OpenVPMSException exception) {
            String title = Messages.get("supplier.order.copy.failed");
            ErrorHelper.show(title, exception);
        }
    }

    /**
     * Returns a {@link ActPoster} to post an act.
     *
     * @param object the act to post
     * @return the {@link ActPoster}
     */
    @Override
    protected ActPoster<FinancialAct> getActPoster(FinancialAct object) {
        HelpContext help = getHelpContext().subtopic("post");
        return new SupplierOrderPoster(object, OrderActions.INSTANCE, getContext(), help);
    }

    /**
     * Invoked when posting of an act is complete, either by saving the act
     * with <em>POSTED</em> status, or invoking {@link #onPost}.
     * <p/>
     * This implementation sends the order to the supplier if they are ESCI-enabled; otherwise it just prints the act.
     *
     * @param act the act
     */
    @Override
    protected void onPosted(FinancialAct act) {
        SupplierRules rules = new SupplierRules(ServiceHelper.getArchetypeService());
        if (rules.getSupplierStockLocation(act) != null) {
            // ESCI is configured for the supplier, so submit the order
            if (submitOrder(act)) {
                InformationDialog.show(Messages.get("supplier.order.sent.title"),
                                       Messages.get("supplier.order.sent.message"));
                scheduleCheckInbox(true); // poll in 30 secs to see if there are any responses
            }
        } else {
            print(act);
        }
    }

    /**
     * Submits the order to the supplier, via ESCI.
     *
     * @param act the order
     * @return {@code true} if the order was submitted successfully
     */
    protected boolean submitOrder(FinancialAct act) {
        boolean submitted = false;
        try {
            OrderServiceAdapter service = ServiceHelper.getOrderService();
            service.submitOrder(act);
            submitted = true;
        } catch (Throwable exception) {
            // failed to submit the order, so revert to IN_PROGRESS
            act.setStatus(ActStatus.IN_PROGRESS);
            SaveHelper.save(act);

            ErrorHelper.show(exception);
            onRefresh(act);
        }
        return submitted;
    }

    /**
     * Invoked to generate orders.
     */
    private void onGenerate() {
        String title = Messages.get("supplier.order.generate.title");
        HelpContext help = getHelpContext().subtopic("generate");
        StockLocationSupplierDialog dialog = new StockLocationSupplierDialog(title, getContext(), help);
        dialog.addWindowPaneListener(new PopupDialogListener() {
            @Override
            public void onOK() {
                Party location = dialog.getStockLocation();
                Party supplier = dialog.getSupplier();
                generateOrders(location, supplier, dialog.getStockLocations(), dialog.getSuppliers(),
                               dialog.getBelowIdealQuantity(), help);
            }
        });
        dialog.show();
    }

    /**
     * Generates orders.
     *
     * @param stockLocation      the selected stock location. May be {@code null} to indicate all stock locations
     * @param supplier           the selected supplier. May be {@code null} to indicate all suppliers
     * @param locations          the available stock locations
     * @param suppliers          the available suppliers
     * @param belowIdealQuantity if {@code true}, generate orders for stock below ideal quantity; else
     *                           generate orders for stock at or below critical quantity
     * @param help               the help context
     */
    private void generateOrders(Party stockLocation, Party supplier, List<IMObject> locations,
                                List<IMObject> suppliers, boolean belowIdealQuantity, HelpContext help) {
        final String title = Messages.get("supplier.order.generate.title");
        if (stockLocation != null) {
            locations = Collections.singletonList(stockLocation);
        }
        if (supplier != null) {
            suppliers = Collections.singletonList(supplier);
        }
        // TODO
        OrderProgressBarProcessor processor = new OrderProgressBarProcessor(
                getContext().getPractice(), locations, suppliers, belowIdealQuantity, title);
        BatchProcessorDialog dialog = new BatchProcessorDialog(processor.getTitle(), processor, help);
        processor.setListener(new BatchProcessorListener() {
            public void completed() {
                dialog.close();
                String message = Messages.format("supplier.order.generated.message", processor.getOrders());
                InformationDialog.show(title, message);
                onRefresh((FinancialAct) null);
            }

            public void error(Throwable exception) {
                ErrorHelper.show(exception);
                onRefresh((FinancialAct) null);
            }
        });
        dialog.show();
    }

}
