/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.supplier.order;

import nextapp.echo2.app.Button;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openvpms.component.business.service.scheduler.JobScheduler;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.esci.adapter.dispatcher.ESCIDispatcher;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.ActActions;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.factory.ButtonFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.supplier.SupplierActCRUDWindow;
import org.quartz.Scheduler;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;

import java.util.Date;
import java.util.List;


/**
 * Supplier CRUD window that adds support to check ESCI inboxes.
 *
 * @author Tim Anderson
 */
public class ESCISupplierCRUDWindow extends SupplierActCRUDWindow<FinancialAct> {

    /**
     * Check inbox button identifier.
     */
    protected static final String CHECK_INBOX_ID = "checkInbox";

    /**
     * The logger.
     */
    private static final Log log = LogFactory.getLog(ESCISupplierCRUDWindow.class);

    /**
     * Constructs an {@link ESCISupplierCRUDWindow}.
     *
     * @param archetypes the archetypes that this may create
     * @param operations determines the operations that may be performed on the selected object
     * @param context    the context
     * @param help       the help context
     */
    public ESCISupplierCRUDWindow(Archetypes<FinancialAct> archetypes, ActActions<FinancialAct> operations,
                                  Context context, HelpContext help) {
        super(archetypes, operations, context, help);
    }

    /**
     * Creates a button that invokes {@link #checkInbox} when pressed.
     *
     * @return a new button
     */
    protected Button createCheckInboxButton() {
        return ButtonFactory.create(CHECK_INBOX_ID, this::checkInbox);
    }

    /**
     * Check the ESCI inboxes for messages from the suppliers.
     */
    protected void checkInbox() {
        ESCIErrorHandler handler = new ESCIErrorHandler();
        ESCIDispatcher dispatcher = ServiceHelper.getBean(ESCIDispatcher.class);
        dispatcher.dispatch(true, handler);
        if (handler.getErrors() != 0) {
            ErrorHelper.show(Messages.format("supplier.esci.checkinbox.error", handler.formatErrors()));
        }
    }

    /**
     * Schedule a poll of the ESCI inboxes to pick up messages.
     *
     * @param delay if {@code true} add a 30-second delay
     */
    protected void scheduleCheckInbox(boolean delay) {
        try {
            String name = getESCIJobName();
            if (!StringUtils.isEmpty(name)) {
                String triggerName = "ESCIDispatcherAdhocTrigger";
                Scheduler scheduler = ServiceHelper.getBean(Scheduler.class);
                scheduler.unscheduleJob(new TriggerKey(triggerName)); // remove the existing trigger, if any

                TriggerBuilder<Trigger> builder = TriggerBuilder.newTrigger().withIdentity(triggerName).forJob(name);
                if (delay) {
                    Date in30secs = new Date(System.currentTimeMillis() + 30 * 1000);
                    builder.startAt(in30secs);
                }
                Trigger trigger = builder.build();
                scheduler.scheduleJob(trigger);
            }
        } catch (Throwable exception) {
            ErrorHelper.show(exception);
        }
    }

    /**
     * Returns the ESCI job configuration name.
     *
     * @return the ESCI job configuration name, or {@code null} if none exists
     */
    private String getESCIJobName() {
        String name = null;
        JobScheduler scheduler = ServiceHelper.getBean(JobScheduler.class);
        List<Entity> jobs = scheduler.getJobs("entity.jobESCIInboxReader");
        if (!jobs.isEmpty()) {
            name = scheduler.getJobName(jobs.get(0));
        }
        return name;
    }

}
