/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.account;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.Label;
import nextapp.echo2.app.Row;
import nextapp.echo2.app.SelectField;
import nextapp.echo2.app.event.ActionEvent;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.practice.Location;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.user.User;
import org.openvpms.component.system.common.query.SortConstraint;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.bound.BoundTextComponentFactory;
import org.openvpms.web.component.im.clinician.ClinicianSelectField;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.location.LocationSelectField;
import org.openvpms.web.component.im.lookup.LookupField;
import org.openvpms.web.component.im.query.AbstractQueryState;
import org.openvpms.web.component.im.query.ActStatuses;
import org.openvpms.web.component.im.query.DateRange;
import org.openvpms.web.component.im.query.DateRangeActQuery;
import org.openvpms.web.component.im.query.QueryState;
import org.openvpms.web.component.im.query.ResultSet;
import org.openvpms.web.component.property.ModifiableListener;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.SimpleProperty;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.event.ActionListener;
import org.openvpms.web.echo.factory.GridFactory;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.factory.RowFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.style.Styles;
import org.openvpms.web.echo.text.TextField;
import org.openvpms.web.echo.util.StyleSheetHelper;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

/**
 * Query account acts by customer name, location, clinician, date and amount.
 *
 * @author Tim Anderson
 */
public abstract class AccountActQuery extends DateRangeActQuery<FinancialAct> {

    /**
     * Determines if a search filter should be displayed.
     */
    private final boolean searchFilter;

    /**
     * Determines if an amount filter should be displayed.
     */
    private final boolean amountFilter;

    /**
     * The location selector.
     */
    private final LocationSelectField locationSelector;

    /**
     * Clinician selector. May be {@code null}
     */
    private final SelectField clinicianSelector;

    /**
     * The amount-from  field.
     */
    private final Property fromAmount = new SimpleProperty("amountFrom", BigDecimal.class);

    /**
     * The amount-to field.
     */
    private final Property toAmount = new SimpleProperty("amountTo", BigDecimal.class);

    /**
     * 0
     * Listener for amountFrom changes.
     */
    private final ModifiableListener fromAmountListener;

    /**
     * Listener for amountTo changes.
     */
    private final ModifiableListener toAmountListener;

    /**
     * The maximum number of months to query when querying by amount.
     */
    private final int dateRangeMonths = 6;

    /**
     * Constructs an {@link AccountActQuery}.
     *
     * @param archetypes      the archetypes to query
     * @param statuses        the statuses to query
     * @param location        the initial location to filter on. May be {@code null}
     * @param searchFilter    if {@code true}, add a search filter
     * @param clinicianFilter if {@code true}, add a clinician filter
     * @param amountFilter    if {@code true}, add an amount filter
     * @param context         the layout context
     */
    public AccountActQuery(String[] archetypes, ActStatuses statuses, Party location, boolean searchFilter,
                           boolean clinicianFilter, boolean amountFilter, LayoutContext context) {
        super(null, null, null, archetypes, statuses, FinancialAct.class);
        this.searchFilter = searchFilter;
        this.amountFilter = amountFilter;
        locationSelector = createLocationSelector(context.getContext(), location);
        clinicianSelector = (clinicianFilter) ? createClinicianSelector() : null;
        if (amountFilter) {
            setDefaultSortConstraint(ASCENDING_START_TIME);

            fromAmountListener = modifiable -> onAmountFromChanged();
            fromAmount.addModifiableListener(fromAmountListener);
            toAmountListener = modifiable -> onAmountToChanged();
            toAmount.addModifiableListener(toAmountListener);
        } else {
            fromAmountListener = null;
            toAmountListener = null;
        }
    }

    /**
     * Sets the selected location.
     *
     * @param location the location. May be {@code null}
     */
    public void setLocation(Party location) {
        locationSelector.setSelected(location != null ? new Location(location) : Location.ALL);
    }

    /**
     * Returns the selected clinician.
     *
     * @return the selected clinician. May be {@code null}
     */
    public User getClinician() {
        return clinicianSelector != null ? (User) clinicianSelector.getSelectedItem() : null;
    }

    /**
     * Sets the clinician.
     *
     * @param clinician the clinician. May be {@code null}
     */
    public void setClinician(User clinician) {
        if (clinicianSelector != null) {
            clinicianSelector.setSelectedItem(clinician);
        }
    }

    /**
     * Returns the preferred height of the query when rendered.
     *
     * @return the preferred height, or {@code null} if it has no preferred height
     */
    @Override
    public Extent getHeight() {
        int height = 2;
        if (searchFilter || amountFilter || clinicianSelector != null) {
            height++;
        }
        return super.getHeight(height);
    }

    /**
     * Performs the query.
     *
     * @param sort the sort constraint. May be {@code null}
     * @return the query result set. May be {@code null}
     */
    @Override
    public ResultSet<FinancialAct> query(SortConstraint[] sort) {
        ResultSet<FinancialAct> result = null;
        if (haveAmount()) {
            DateRange range = getDateRange();
            if (range.getAllDates()) {
                ErrorHelper.show(Messages.get("reporting.account.search.dateRangeRequired"));
            } else if (range.getFrom() == null || range.getTo() == null
                       || calculateFromDate(range.getTo()).compareTo(range.getFrom()) > 0) {
                ErrorHelper.show(Messages.format("reporting.account.search.dateRangeTooBig", dateRangeMonths));
            } else {
                result = createResultSet(sort);
            }
        } else {
            result = createResultSet(sort);
        }
        return result;
    }

    /**
     * Creates a container component to lay out the query component in.
     *
     * @return a new container
     * @see #doLayout(Component)
     */
    @Override
    protected Component createContainer() {
        return GridFactory.create(6);
    }

    /**
     * Lays out the component in a container.
     *
     * @param container the container
     */
    @Override
    protected void doLayout(Component container) {
        if (searchFilter) {
            addSearchField(container);
        }
        addShortNameSelector(container);
        addStatusSelector(container);
        addDateRange(container);
        addLocation(container);
        if (clinicianSelector != null) {
            addClinician(container);
        }
        if (amountFilter) {
            addAmount(container);
        }
    }

    /**
     * Creates a new result set.
     *
     * @param sort the sort constraint. May be {@code null}
     * @return a new result set
     */
    @Override
    protected ResultSet<FinancialAct> createResultSet(SortConstraint[] sort) {
        return new AccountActResultSet(getArchetypeConstraint(), getValue(), getLocation(), getLocations(),
                                       getFrom(), getTo(), getClinician(), fromAmount.getBigDecimal(),
                                       toAmount.getBigDecimal(), getStatuses(), excludeStatuses(), getMaxResults(),
                                       sort);
    }

    /**
     * Returns the available locations.
     *
     * @return the available locations
     */
    protected List<Party> getLocations() {
        return locationSelector.getLocations();
    }

    /**
     * Returns the selected location.
     *
     * @return the selected location. May be {@code null}
     */
    protected Party getLocation() {
        return (Party) locationSelector.getSelectedItem();
    }

    /**
     * Creates a field to select the location.
     *
     * @param context  the context
     * @param location the initial location. May  be {@code null}
     * @return a new selector
     */
    protected LocationSelectField createLocationSelector(Context context, Party location) {
        LocationSelectField result = new LocationSelectField(context.getUser(), context.getPractice(), true);
        if (location != null) {
            result.setSelectedItem(location);
        }
        result.addActionListener(new ActionListener() {
            @Override
            public void onAction(ActionEvent event) {
                onQuery();
            }
        });
        return result;
    }

    /**
     * Adds the location selector to a container.
     *
     * @param container the container
     */
    protected void addLocation(Component container) {
        Label label = LabelFactory.create();
        label.setText(DescriptorHelper.getDisplayName(CustomerAccountArchetypes.INVOICE, "location",
                                                      ServiceHelper.getArchetypeService()));
        container.add(label);
        container.add(locationSelector);
        getFocusGroup().add(locationSelector);
    }

    /**
     * Adds the clinician selector to a container, if querying by clinician.
     *
     * @param container the container
     */
    protected void addClinician(Component container) {
        Label label = LabelFactory.create();
        label.setText(Messages.get("label.clinician"));
        container.add(label);
        container.add(clinicianSelector);
        getFocusGroup().add(clinicianSelector);
    }

    /**
     * Adds the amount range.
     *
     * @param container the container to add to
     */
    protected void addAmount(Component container) {
        String amount = DescriptorHelper.getDisplayName(CustomerAccountArchetypes.INVOICE, "amount",
                                                        ServiceHelper.getArchetypeService());
        Row amountRange = RowFactory.create(Styles.CELL_SPACING);
        int numericLength = StyleSheetHelper.getNumericLength();
        TextField from = BoundTextComponentFactory.createNumeric(fromAmount, numericLength);
        TextField to = BoundTextComponentFactory.createNumeric(toAmount, numericLength);
        amountRange.add(from);
        amountRange.add(LabelFactory.text("-"));
        amountRange.add(to);
        container.add(LabelFactory.text(amount));
        container.add(amountRange);
        FocusGroup group = getFocusGroup();
        group.add(from);
        group.add(to);
    }

    /**
     * Returns the query state.
     *
     * @return the query state
     */
    @Override
    public QueryState getQueryState() {
        return new Memento(this);
    }

    /**
     * Sets the query state.
     *
     * @param state the query state
     */
    @Override
    public void setQueryState(QueryState state) {
        if (state instanceof Memento) {
            Memento memento = (Memento) state;
            setShortName(memento.archetype);
            setStatus(memento.status);
            getDateRange().setAllDates(memento.all);
            getDateRange().setFrom(memento.from);
            getDateRange().setTo(memento.to);
            setLocation(memento.location);
            setClinician(memento.clinician);
            fromAmount.setValue(memento.fromAmount);
            toAmount.setValue(memento.toAmount);
        }
    }

    private static class Memento extends AbstractQueryState {

        private final String archetype;

        private final String status;

        private final boolean all;

        private final Date from;

        private final Date to;

        private final Party location;

        private final User clinician;

        private final BigDecimal fromAmount;

        private final BigDecimal toAmount;

        /**
         * Constructs a {@link Memento}.
         *
         * @param query the query
         */
        public Memento(AccountActQuery query) {
            super(query);
            this.archetype = query.getShortName();
            LookupField status = query.getStatusSelector();
            this.status = (status != null) ? status.getSelectedCode() : null;
            this.all = query.getAllDates();
            this.from = query.getFrom();
            this.to = query.getTo();
            this.location = query.getLocation();
            this.clinician = query.getClinician();
            this.fromAmount = query.fromAmount.getBigDecimal();
            this.toAmount = query.toAmount.getBigDecimal();
        }
    }


    /**
     * Creates a new dropdown to select clinicians.
     *
     * @return a new clinician selector
     */
    private SelectField createClinicianSelector() {
        SelectField result = new ClinicianSelectField();
        result.addActionListener(new ActionListener() {
            public void onAction(ActionEvent event) {
                onQuery();
            }
        });
        return result;
    }

    /**
     * Invoked when the 'to' amount changes.
     */
    private void onAmountToChanged() {
        BigDecimal from = fromAmount.getBigDecimal();
        BigDecimal to = toAmount.getBigDecimal();
        if (to != null) {
            getDateRange().setAllDates(false);
            if (from == null || to.compareTo(from) < 0) {
                fromAmount.removeModifiableListener(fromAmountListener);
                fromAmount.setValue(to);
                fromAmount.addModifiableListener(fromAmountListener);
            }
        }
    }

    /**
     * Invoked when the 'from' amount changes.
     */
    private void onAmountFromChanged() {
        BigDecimal from = fromAmount.getBigDecimal();
        BigDecimal to = toAmount.getBigDecimal();
        if (from != null) {
            getDateRange().setAllDates(false);
            if (to == null || to.compareTo(from) < 0) {
                toAmount.removeModifiableListener(toAmountListener);
                toAmount.setValue(from);
                toAmount.addModifiableListener(toAmountListener);
            }
        }
    }

    /**
     * Calculates the 'from' date.
     *
     * @param to the 'to' date
     * @return the new 'from' date
     */
    private Date calculateFromDate(Date to) {
        return DateRules.getDate(to, -dateRangeMonths, DateUnits.MONTHS);
    }

    /**
     * Determines if an amount has been specified.
     *
     * @return {@code true} if an amount has been specified
     */
    private boolean haveAmount() {
        return fromAmount.getBigDecimal() != null || toAmount.getBigDecimal() != null;
    }

}
