/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.product.stock;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.Row;
import org.openvpms.archetype.rules.stock.StockArchetypes;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.SortConstraint;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.query.ActStatuses;
import org.openvpms.web.component.im.query.DateRangeActQuery;
import org.openvpms.web.component.im.query.ResultSet;
import org.openvpms.web.component.im.select.AbstractSelectorListener;
import org.openvpms.web.component.im.select.IMObjectSelector;
import org.openvpms.web.echo.factory.ColumnFactory;
import org.openvpms.web.echo.factory.RowFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.style.Styles;


/**
 * Query for <em>act.stock*</em> acts.
 *
 * @author Tim Anderson
 */
public class StockQuery extends DateRangeActQuery<Act> {

    /**
     * The stock location selector.
     */
    private final IMObjectSelector<Party> stockLocation;

    /**
     * The act statuses.
     */
    private static final ActStatuses STATUSES = new ActStatuses(StockArchetypes.STOCK_TRANSFER);


    /**
     * Constructs a {@link StockQuery}.
     *
     * @param shortNames the act short names to query
     * @param context    the context
     * @param help       the help context
     */
    public StockQuery(String[] shortNames, Context context, HelpContext help) {
        super(shortNames, STATUSES, Act.class);

        setParticipantConstraint(null, "stockLocation", StockArchetypes.STOCK_LOCATION_PARTICIPATION);

        stockLocation = new StockLocationSelector(new DefaultLayoutContext(context, help));
        stockLocation.setListener(new AbstractSelectorListener<Party>() {
            public void selected(Party object) {
                setEntity(object);
                onQuery();
            }
        });

        setEntity(stockLocation.getObject());
    }

    /**
     * Performs the query.
     *
     * @param sort the sort constraint. May be {@code null}
     * @return the query result set
     * @throws ArchetypeServiceException if the query fails
     */
    @Override
    public ResultSet<Act> query(SortConstraint[] sort) {
        return createResultSet(sort);
    }

    /**
     * Returns the preferred height of the query when rendered.
     *
     * @return the preferred height, or {@code null} if it has no preferred height
     */
    @Override
    public Extent getHeight() {
        return getHeight(2);
    }

    /**
     * Lays out the component in a container.
     *
     * @param container the container
     */
    @Override
    protected void doLayout(Component container) {
        Row row1 = RowFactory.create(Styles.CELL_SPACING);
        Row row2 = RowFactory.create(Styles.CELL_SPACING);

        addShortNameSelector(row1);
        row1.add(stockLocation.getComponent());
        addStatusSelector(row1);
        addDateRange(row2);

        container.add(ColumnFactory.create(Styles.CELL_SPACING, row1, row2));
    }
}
