/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.patient.problem;

import nextapp.echo2.app.Column;
import nextapp.echo2.app.Label;
import nextapp.echo2.app.RadioButton;
import nextapp.echo2.app.button.ButtonGroup;
import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.archetype.rules.prefs.PreferenceArchetypes;
import org.openvpms.archetype.rules.prefs.Preferences;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.business.service.archetype.rule.IArchetypeRuleService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.object.Relationship;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.component.im.edit.EditDialog;
import org.openvpms.web.component.im.edit.IMObjectEditor;
import org.openvpms.web.component.im.print.IMObjectReportPrinter;
import org.openvpms.web.component.im.print.IMPrinterFactory;
import org.openvpms.web.component.im.print.InteractiveIMPrinter;
import org.openvpms.web.component.im.report.ContextDocumentTemplateLocator;
import org.openvpms.web.component.im.report.DocumentTemplateLocator;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.im.util.LookupNameHelper;
import org.openvpms.web.component.retry.Retryer;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.dialog.ConfirmationDialog;
import org.openvpms.web.echo.dialog.PopupDialog;
import org.openvpms.web.echo.dialog.PopupDialogListener;
import org.openvpms.web.echo.factory.ButtonFactory;
import org.openvpms.web.echo.factory.ColumnFactory;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.style.Styles;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.resource.i18n.format.DateFormatter;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.patient.PatientMedicalRecordLinker;
import org.openvpms.web.workspace.patient.history.AbstractPatientHistoryCRUDWindow;
import org.openvpms.web.workspace.patient.history.PatientHistoryActions;
import org.openvpms.web.workspace.patient.history.TextSearch;

import static org.openvpms.component.system.common.query.Constraints.eq;
import static org.openvpms.component.system.common.query.Constraints.join;
import static org.openvpms.component.system.common.query.Constraints.shortName;


/**
 * CRUD Window for patient record acts in 'problem' view.
 *
 * @author Tim Anderson
 */
public class ProblemRecordCRUDWindow extends AbstractPatientHistoryCRUDWindow {

    /**
     * The current problem.
     */
    private Act problem;

    /**
     * The query, for printing.
     */
    private ProblemQuery query;


    /**
     * Constructs a {@link ProblemRecordCRUDWindow}.
     *
     * @param context the context
     * @param help    the help context
     */
    public ProblemRecordCRUDWindow(Context context, HelpContext help) {
        super(Archetypes.create(PatientArchetypes.CLINICAL_PROBLEM, Act.class,
                                Messages.get("patient.record.createtype")),
              ProblemActions.INSTANCE, context, help);
    }

    /**
     * Sets the object.
     *
     * @param object the object. May be {@code null}
     */
    @Override
    public void setObject(Act object) {
        super.setObject(object);
        if (object != null) {
            if (TypeHelper.isA(object, PatientArchetypes.CLINICAL_PROBLEM)) {
                setProblem(object);
            } else if (TypeHelper.isA(object, PatientArchetypes.CLINICAL_EVENT)) {
                setProblem(null); // can't determine which problem to use.
            } else {
                setProblem(getSource(object, PatientArchetypes.CLINICAL_PROBLEM));
            }
        }
    }

    /**
     * Sets the current patient clinical problem.
     *
     * @param problem the current problem. May be {@code null}
     */
    public void setProblem(Act problem) {
        this.problem = problem;
    }

    /**
     * Returns the current patient clinical problem.
     *
     * @return the current problem. May be {@code null}
     */
    public Act getProblem() {
        return problem;
    }

    /**
     * Sets the current query, for printing.
     *
     * @param query the query
     */
    public void setQuery(ProblemQuery query) {
        this.query = query;
    }

    /**
     * Determines the actions that may be performed on the selected object.
     *
     * @return the actions
     */
    @Override
    protected ProblemActions getActions() {
        return (ProblemActions) super.getActions();
    }

    /**
     * Invoked when the 'print' button is pressed.
     * This implementation prints the current summary list, rather than
     * the selected item.
     */
    @Override
    protected void onPrint() {
        if (query != null) {
            Context context = getContext();
            IMObjectReportPrinter<Act> printer = createPrinter(context);
            String title = Messages.get("patient.record.problem.print");
            HelpContext help = getHelpContext().topic(PatientArchetypes.CLINICAL_PROBLEM + "/print");
            InteractiveIMPrinter<Act> iPrinter = new InteractiveIMPrinter<>(title, printer, context, help);
            iPrinter.setMailContext(getMailContext());
            iPrinter.print();
        }
    }

    /**
     * Invoked to preview the problem history.
     */
    @Override
    protected void onPreview() {
        if (query != null) {
            Context context = getContext();
            IMObjectReportPrinter<Act> printer = createPrinter(context);
            preview(printer);
        }
    }

    /**
     * Invoked when the 'mail' button is pressed.
     */
    @Override
    protected void onMail() {
        if (query != null) {
            Context context = getContext();
            IMObjectReportPrinter<Act> printer = createPrinter(context);
            mail(printer);
        }
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        super.layoutButtons(buttons);
        buttons.add(createMarkReviewedButton());
        buttons.add(createUnmarkReviewedButton());
        buttons.add(createExternalEditButton());
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        super.enableButtons(buttons, enable);
        enablePrintPreview(buttons, enable);
    }

    /**
     * Invoked when the 'new' button is pressed.
     *
     * @param archetypes the archetypes
     */
    @Override
    protected void onCreate(Archetypes<Act> archetypes) {
        if (problem != null) {
            boolean includeAddendum = false;
            String defaultShortName = PatientArchetypes.CLINICAL_NOTE;
            Act selected = getObject();
            if (TypeHelper.isA(selected, PatientArchetypes.CLINICAL_NOTE, PatientArchetypes.PATIENT_MEDICATION)) {
                includeAddendum = true;
                if (getActions().isLocked(selected)) {
                    defaultShortName = PatientArchetypes.CLINICAL_ADDENDUM;
                }
            }
            // problem is selected, so display all of the possible event item archetypes
            String[] shortNames = getShortNames(PatientArchetypes.CLINICAL_PROBLEM_ITEM,
                                                includeAddendum, PatientArchetypes.CLINICAL_PROBLEM);
            archetypes = new Archetypes<>(shortNames, archetypes.getType(), defaultShortName,
                                          archetypes.getDisplayName());
        }
        super.onCreate(archetypes);
    }

    /**
     * Invoked when a new object has been created.
     * <p>
     * If the object is an <em>act.patientClinicalProblem</em>, a dialog will be displayed prompting for the visit
     * to link it to.
     *
     * @param object the new object
     */
    @Override
    protected void onCreated(Act object) {
        if (object.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
            Act event = getLatestEvent(getContext().getPatient());
            if (event != null) {
                // there is an existing event for the patient. Prompt to add the problem to this visit, or a new one.
                VisitSelectionDialog dialog = new VisitSelectionDialog(event, getHelpContext());
                dialog.addWindowPaneListener(new PopupDialogListener() {
                    @Override
                    public void onOK() {
                        if (dialog.createVisit()) {
                            createEvent();
                        } else {
                            setEvent(event);
                        }
                        ProblemRecordCRUDWindow.super.onCreated(object);
                    }
                });
                dialog.show();
            } else {
                // there is an no event for the patient. Prompt to create a new one.
                ConfirmationDialog dialog = new ConfirmationDialog(
                        Messages.get("patient.record.problem.createVisit.title"),
                        Messages.get("patient.record.problem.createVisit.message"));
                dialog.addWindowPaneListener(new PopupDialogListener() {
                    @Override
                    public void onOK() {
                        createEvent();
                        ProblemRecordCRUDWindow.super.onCreated(object);
                    }
                });
                dialog.show();
            }
        } else {
            super.onCreated(object);
        }
    }

    /**
     * Creates a new edit dialog.
     * <p>
     * This implementation registers a post save callback to link the act into the patient history on save.<br/>
     * This is needed for dialogs that show patient history, in order for new records to be displayed when Apply is
     * pressed.
     *
     * @param editor the editor
     * @return a new edit dialog
     */
    @Override
    protected EditDialog createEditDialog(IMObjectEditor editor) {
        EditDialog dialog = super.createEditDialog(editor);
        dialog.setPostSaveCallback(e -> linkRecords((Act) e.getObject()));
        return dialog;
    }

    /**
     * Links patient records, once they have been saved.
     *
     * @param act the act
     */
    protected void linkRecords(Act act) {
        Act problem;
        if (!act.isA(PatientArchetypes.CLINICAL_EVENT)) {
            if (!act.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
                problem = getProblem(act);
            } else {
                problem = act;
                act = null;
            }
            Act event = getEvent();
            PatientMedicalRecordLinker linker;
            Act selected = getObject();
            if (TypeHelper.isA(act, PatientArchetypes.CLINICAL_ADDENDUM)) {
                if (!TypeHelper.isA(selected, PatientArchetypes.CLINICAL_ADDENDUM)) {
                    linker = createMedicalRecordLinker(event, problem, selected, act);
                } else {
                    linker = createMedicalRecordLinker(event, problem, null, act);
                }
            } else {
                linker = createMedicalRecordLinker(event, problem, act, null);
            }
            Retryer.run(linker);
        }
    }

    /**
     * Invoked when the object has been deleted.
     *
     * @param object the object
     */
    @Override
    protected void onDeleted(Act object) {
        if (TypeHelper.isA(object, PatientArchetypes.CLINICAL_PROBLEM)) {
            setProblem(null);
        }
        super.onDeleted(object);
    }

    /**
     * Creates a printer to print/preview problems.
     *
     * @param context the context
     * @return a new printer
     */
    private IMObjectReportPrinter<Act> createPrinter(Context context) {
        TextSearch search = null;
        String value = query.getValue();
        Preferences preferences = ServiceHelper.getPreferences();
        IArchetypeRuleService service = ServiceHelper.getArchetypeService();
        if (!StringUtils.isEmpty(value)) {
            boolean showClinician = preferences.getBoolean(PreferenceArchetypes.HISTORY, "showClinician", false);
            boolean showBatches = preferences.getBoolean(PreferenceArchetypes.HISTORY, "showBatches", false);
            search = new TextSearch(value, showClinician, showBatches, service);
        }
        ProblemFilter filter = new ProblemFilter(query.getSelectedArchetypes(), search, query.isSortAscending(),
                                                 service);
        Iterable<Act> summary = new ProblemHierarchyIterator(query, filter);
        DocumentTemplateLocator locator = new ContextDocumentTemplateLocator(PatientArchetypes.CLINICAL_PROBLEM,
                                                                             context);
        IMPrinterFactory factory = ServiceHelper.getBean(IMPrinterFactory.class);
        return factory.createIMObjectReportPrinter(summary, locator, context);
    }

    /**
     * Returns the latest event for a patient.
     *
     * @param patient the patient
     * @return the latest event, or {@code null} if none is found
     */
    private Act getLatestEvent(Party patient) {
        ArchetypeQuery query = new ArchetypeQuery(shortName("e", PatientArchetypes.CLINICAL_EVENT));
        query.add(join("patient").add(eq("entity", patient)));
        query.add(Constraints.sort("startTime", false));
        query.setMaxResults(1);
        IMObjectQueryIterator<Act> iterator = new IMObjectQueryIterator<>(query);
        return (iterator.hasNext()) ? iterator.next() : null;
    }

    /**
     * Returns the problem associated with an act.
     * <p>
     * If the act has an associated problem, this will be returned, otherwise the current {@link #problem} will
     * be returned.
     *
     * @param act the act
     * @return the associated event, or {@code null} if none is found and there is no current event
     */
    private Act getProblem(Act act) {
        Act result = getSource(act, PatientArchetypes.CLINICAL_PROBLEM);
        return result != null ? result : problem;
    }

    /**
     * Returns the source of an act with the specified short name.
     *
     * @param act       the act
     * @param shortName the archetype short name
     * @return the source, or {@code null} if none exists
     */
    private Act getSource(Act act, String shortName) {
        for (Relationship relationship : act.getTargetActRelationships()) {
            Reference source = relationship.getSource();
            if (source.isA(shortName)) {
                return (Act) IMObjectHelper.getObject(source, getContext());
            }
        }
        return null;
    }

    private static class ProblemActions extends PatientHistoryActions {

        public static final ProblemActions INSTANCE = new ProblemActions();

        /**
         * Determines if an act can be deleted.
         *
         * @param act the act to check
         * @return {@code true} if the act can be deleted
         */
        @Override
        public boolean canDelete(Act act) {
            return !TypeHelper.isA(act, PatientArchetypes.CLINICAL_EVENT) && super.canDelete(act);
        }
    }

    private static class VisitSelectionDialog extends PopupDialog {

        /**
         * If selected, indicates to create a new visit.
         */
        private final RadioButton newVisit;

        /**
         * Constructs a {@link VisitSelectionDialog}.
         *
         * @param event the existing event
         * @param help  the help context
         */
        public VisitSelectionDialog(Act event, HelpContext help) {
            super(Messages.get("patient.record.problem.selectVisit.title"), OK_CANCEL, help.subtopic("selectVisit"));
            setModal(true);
            ButtonGroup group = new ButtonGroup();
            Runnable listener = () -> {
                // no-op
            };
            RadioButton existingVisit = ButtonFactory.create(null, group, listener);
            existingVisit.setText(getEventDescription(event));
            existingVisit.setSelected(true);
            newVisit = ButtonFactory.create("patient.record.problem.selectVisit.new", group, listener);
            existingVisit.setGroup(group);
            newVisit.setGroup(group);
            Label label = LabelFactory.create("patient.record.problem.selectVisit.message");
            Column column = ColumnFactory.create(Styles.CELL_SPACING, label, existingVisit, newVisit);
            getLayout().add(ColumnFactory.create(Styles.LARGE_INSET, column));
        }

        /**
         * Determines if a new visit should be created.
         *
         * @return if {@code true}, create a new visit, otherwise use the existing one
         */
        public boolean createVisit() {
            return newVisit.isSelected();
        }

        /**
         * Formats a clinical event description.
         *
         * @param act the event
         * @return the description
         */
        private String getEventDescription(Act act) {
            String result;
            String title = act.getTitle();
            String reason = LookupNameHelper.getName(act, "reason");
            String date = DateFormatter.formatDate(act.getActivityStartTime(), false);
            if (!StringUtils.isEmpty(reason) && !StringUtils.isEmpty(title)) {
                result = Messages.format("patient.record.problem.selectVisit.datedReasonAndSummary", date, reason,
                                         title);
            } else if (!StringUtils.isEmpty(reason)) {
                result = Messages.format("patient.record.problem.selectVisit.datedReason", date, reason);
            } else if (!StringUtils.isEmpty(title)) {
                result = Messages.format("patient.record.problem.selectVisit.datedReason", date, title);
            } else {
                result = date;
            }
            return result;
        }
    }
}
