/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.event.WindowPaneEvent;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.system.common.util.Variables;
import org.openvpms.paymentprocessor.exception.PaymentProcessorException;
import org.openvpms.paymentprocessor.internal.service.PaymentProcessorServiceAdapter;
import org.openvpms.paymentprocessor.internal.service.PaymentProcessors;
import org.openvpms.paymentprocessor.internal.transaction.PaymentProcessorTransactionFactory;
import org.openvpms.paymentprocessor.processor.TransactionMode;
import org.openvpms.paymentprocessor.service.PaymentProcessorService;
import org.openvpms.paymentprocessor.service.PaymentRequirements;
import org.openvpms.paymentprocessor.service.RefundRequirements;
import org.openvpms.paymentprocessor.service.TransactionRequirements;
import org.openvpms.paymentprocessor.service.TransactionRequirements.Field;
import org.openvpms.paymentprocessor.service.ValidationStatus;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.paymentprocessor.transaction.Transaction.Status;
import org.openvpms.web.component.im.delete.Deletable;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.property.MutableProperty;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.PropertySet;
import org.openvpms.web.component.property.PropertySetBuilder;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.dialog.ConfirmationDialog;
import org.openvpms.web.echo.event.WindowPaneListener;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.resource.i18n.format.NumberFormatter;
import org.openvpms.web.system.ServiceHelper;

import java.math.BigDecimal;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;


/**
 * An editor for <em>act.customerAccountPaymentPP</em> and <em>act.customerAccountRefundPP</em>.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorPaymentItemEditor extends TransactionPaymentItemEditor {

    /**
     * The transaction factory.
     */
    private final PaymentProcessorTransactionFactory transactionFactory;

    /**
     * The expected amount, if this is a reversal.
     */
    private final BigDecimal reversalAmount;

    /**
     * The payment editor that manages this.
     */
    private AbstractCustomerPaymentEditor paymentEditor;

    /**
     * Determines if requirements need to be refreshed.
     */
    private boolean refreshRequirements = true;

    /**
     * The payment requirements.
     */
    private PaymentRequirements paymentRequirements;

    /**
     * The refund requirements.
     */
    private RefundRequirements refundRequirements;

    /**
     * Determines if properties should be marked read-only. This should be the case when the last transaction
     * is IN_PROGRESS, SUBMITTED, or COMPLETED.
     */
    private boolean readOnly;

    /**
     * Payment processor node.
     */
    private static final String PAYMENT_PROCESSOR = "paymentProcessor";

    /**
     * Transaction description node.
     */
    private static final String DESCRIPTION = "description";

    /**
     * Customer email node.
     */
    private static final String EMAIL = "email";

    /**
     * The transaction mode node.
     */
    private static final String TRANSACTION_MODE = "transactionMode";

    /**
     * Constructs {@link PaymentProcessorPaymentItemEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent act
     * @param context the layout context
     */
    public PaymentProcessorPaymentItemEditor(FinancialAct act, FinancialAct parent, LayoutContext context) {
        super(act, parent, context);
        transactionFactory = ServiceHelper.getBean(PaymentProcessorTransactionFactory.class);
        if (act.isA(REFUND_PP)) {
            FinancialAct reversal = getBean(act).getSource("reverses", FinancialAct.class);
            reversalAmount = (reversal != null) ? reversal.getTotal() : null;
        } else {
            reversalAmount = null;
        }
        if (getTransactionMode() == null) {
            // for now, the only mode supported is LINK.
            setTransactionMode(TransactionMode.LINK);
        }
        updateTransactionStatus();
    }

    /**
     * Sets the editor that manages this payment item.
     * <p/>
     * This is used to set the payment status to IN_PROGRESS if it is POSTED, when the item is incomplete.
     *
     * @param paymentEditor the editor
     */
    public void setPaymentEditor(AbstractCustomerPaymentEditor paymentEditor) {
        this.paymentEditor = paymentEditor;
    }

    /**
     * Returns the payment processor.
     *
     * @return the payment processor. May be {@code null}
     */
    public Entity getPaymentProcessor() {
        return getParticipant(PAYMENT_PROCESSOR);
    }

    /**
     * Sets the payment processor.
     *
     * @param paymentProcessor the payment processor
     */
    public void setPaymentProcessor(Entity paymentProcessor) {
        setParticipant(PAYMENT_PROCESSOR, paymentProcessor);
        paymentRequirements = null;
        refundRequirements = null;
        refreshRequirements = true;
    }

    /**
     * Returns a display name for the object being edited.
     *
     * @return a display name for the object
     */
    @Override
    public String getDisplayName() {
        Entity paymentProcessor = getPaymentProcessor();
        return paymentProcessor != null ? paymentProcessor.getName() : super.getDisplayName();
    }

    /**
     * Determines if this item can appear in a split payment or refund.
     * <p/>
     * Split transactions for payment processor items aren't supported, as reversals are difficult to implement.
     *
     * @return {@code false}
     */
    @Override
    public boolean supportsSplitTransactions() {
        return false;
    }

    /**
     * Determines if the current transaction can be cancelled.
     *
     * @return {@code true} if current transaction can be cancelled, otherwise {@code false}
     */
    @Override
    public boolean canCancelTransaction() {
        boolean result = false;
        if (!isComplete()) {
            FinancialAct transaction = getLastTransaction();
            result = transaction != null && canCancel(transaction);
        }
        return result;
    }

    /**
     * Cancels the current transaction.
     */
    @Override
    public void cancelTransaction() {
        FinancialAct transaction = getLastTransaction();
        if (transaction != null && canCancel(transaction)) {
            cancel(transaction);
        }
        onLayout();
    }

    /**
     * Returns the transaction mode.
     *
     * @return the transaction mode. May be {@code null}
     */
    public TransactionMode getTransactionMode() {
        String result = getProperty(TRANSACTION_MODE).getString();
        return result != null ? TransactionMode.valueOf(result) : null;
    }

    /**
     * Sets the transaction mode.
     *
     * @param mode the transaction mode
     */
    public void setTransactionMode(TransactionMode mode) {
        getProperty(TRANSACTION_MODE).setValue((mode != null) ? mode.toString() : null);
    }

    /**
     * Returns the rendered object.
     *
     * @return the rendered object
     */
    @Override
    public Component getComponent() {
        if (refreshRequirements() && getView().hasComponent()) {
            // payment service requirements have changed, so force a layout change
            onLayout();
        }
        return super.getComponent();
    }

    /**
     * Validates the object.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        return super.doValidation(validator)
               && validateUnmatchedRefund(validator)
               && validateReversalAmount(validator);
    }

    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        return new PaymentProcessorPaymentItemLayoutStrategy();
    }

    /**
     * Creates the property set.
     *
     * @param object    the object being edited
     * @param archetype the object archetype
     * @param variables the variables for macro expansion. May be {@code null}
     * @return the property set
     */
    @Override
    protected PropertySet createPropertySet(IMObject object, ArchetypeDescriptor archetype, Variables variables) {
        return new PropertySetBuilder(object, archetype, variables)
                .mutable(TRANSACTION_MODE)
                .mutable(AMOUNT)
                .mutable(DESCRIPTION)
                .mutable(EMAIL)
                .build();
    }

    /**
     * Updates the transaction status.
     *
     * @param transaction the last transaction. May be {@code null}
     * @param complete    if {@code true}, payment/refund is complete, and no further transactions will be performed
     * @return {@code true} if the layout needs to be updated, otherwise {@code false}
     */
    @Override
    protected boolean updateTransactionStatus(FinancialAct transaction, boolean complete) {
        boolean changed = super.updateTransactionStatus(transaction, complete);
        boolean mark = readOnly;
        if (transaction != null) {
            String status = transaction.getStatus();
            readOnly = PaymentProcessorTransactionStatus.IN_PROGRESS.equals(status)
                       || PaymentProcessorTransactionStatus.SUBMITTED.equals(status)
                       || PaymentProcessorTransactionStatus.COMPLETED.equals(status);
        } else {
            readOnly = complete;
        }
        boolean reversal = isReversal();
        // cannot change amount or description for reversals. Email can be changed if the transaction hasn't been
        // successfully submitted
        changed |= mark != readOnly;
        changed |= updateReadOnly(TRANSACTION_MODE, readOnly);
        changed |= updateReadOnly(AMOUNT, readOnly || reversal);
        changed |= updateReadOnly(DESCRIPTION, readOnly || reversal);
        changed |= updateReadOnly(EMAIL, readOnly);
        return changed;
    }

    /**
     * Determines if a transaction allows or prevents deletion of this payment item.
     *
     * @return {@link Deletable#yes()} if the payment item can be deleted, otherwise {@link Deletable#no}
     */
    @Override
    protected Deletable getDeletable(FinancialAct transaction) {
        Deletable result;
        Status status = Status.valueOf(transaction.getStatus());
        if (status == Status.PENDING || status == Status.IN_PROGRESS || status == Status.SUBMITTED
            || status == Status.COMPLETED) {
            String reason;
            if (status == Status.COMPLETED) {
                reason = Messages.format("customer.payment.delete.completedPP", getDisplayName());
            } else {
                reason = Messages.format("customer.payment.delete.outstandingPP", getDisplayName());
            }
            result = Deletable.no(reason);
        } else {
            result = Deletable.yes();
        }
        return result;
    }

    /**
     * Returns the transaction display name, for use in error messages.
     *
     * @return the transaction display name
     */
    @Override
    protected String getTransactionDisplayName() {
        Entity paymentProcessor = getPaymentProcessor();
        return paymentProcessor != null ? paymentProcessor.getName() : "<unknown>";
    }

    /**
     * Performs a transaction.
     * <p/>
     * This is only invoked if the item is valid.
     *
     * @param state    the transaction state
     * @param listener the listener to notify on success
     */
    @Override
    protected void performTransaction(TransactionState state, Runnable listener) {
        Entity paymentProcessor = getPaymentProcessor();
        PaymentProcessorServiceAdapter service = getServiceAdapter(paymentProcessor);
        FinancialAct act = state.getAct();
        if (act == null) {
            act = createTransactionAct(paymentProcessor, state.getSequence());
        }
        Transaction transaction = transactionFactory.getTransaction(act);
        if (transaction.getStatus() == Status.PENDING || transaction.getStatus() == Status.IN_PROGRESS) {
            prepareAndSubmit(transaction, act, service, listener);
        } else if (transaction.getStatus() == Status.SUBMITTED) {
            check(transaction, act, service, listener);
        } else {
            throw new IllegalStateException("Cannot call performTransaction with transaction status="
                                            + transaction.getStatus());
        }
    }

    /**
     * Determines if a new transaction is required.
     *
     * @param transaction the last transaction
     * @return {@code true} if a new transaction is required, otherwise {@code false}
     */
    @Override
    protected boolean isNewTransactionRequired(Act transaction) {
        Status status = Status.valueOf(transaction.getStatus());
        return status == Status.ERROR || status == Status.CANCELLED;
    }

    /**
     * Determines if a transaction is complete. If it is complete, no further transactions may be performed.
     *
     * @param transaction the transaction
     * @return {@code true} if the transaction is complete, otherwise {@code false}
     */
    @Override
    protected boolean isComplete(FinancialAct transaction) {
        Status status = Status.valueOf(transaction.getStatus());
        return status == Status.COMPLETED;
    }

    /**
     * Returns the payment processor service.
     *
     * @param config the payment processor configuration
     * @return the corresponding service
     * @throws PaymentProcessorException if the service is unavailable
     */
    protected PaymentProcessorService getPaymentProcessorService(Entity config) {
        PaymentProcessors paymentProcessors = ServiceHelper.getBean(PaymentProcessors.class);
        return paymentProcessors.getPaymentProcessor(config);
    }

    /**
     * Determines if properties should be marked read-only.
     * <p/>
     * The amount and description should be read-only if a transaction is IN_PROGRESS, SUBMITTED, or COMPLETED.<br/>
     * They must be read-only for IN_PROGRESS transactions to ensure that the user can't change amounts if
     * a submission fails, but the provider recorded the transaction.
     *
     * @return {@code true} if properties should be marked read-only, otherwise {@code false}
     */
    @Override
    protected boolean markPropertiesReadOnly() {
        return readOnly;
    }

    /**
     * Refreshes the layout.
     * <p/>
     * This implementation invokes {@link #updateTransactionStatus()} first, to determine if fields should be read-only
     */
    @Override
    protected void refreshLayout() {
        updateTransactionStatus();
        refreshRequirements();
        onLayout();
    }

    /**
     * Refreshes the transaction population requirements if required.
     *
     * @return {@code true} if the layout needs to be refreshed, otherwise {@code false}
     */
    private boolean refreshRequirements() {
        boolean needsLayout = false;
        if (refreshRequirements) {
            Entity paymentProcessor = getPaymentProcessor();
            if (paymentProcessor != null) {
                try {
                    PaymentProcessorServiceAdapter adapter = getServiceAdapter(paymentProcessor);
                    boolean readOnly = markPropertiesReadOnly();
                    boolean payment = getObject().isA(PAYMENT_PP);
                    TransactionMode mode = getTransactionMode();
                    if (mode != null) {
                        paymentRequirements = (payment) ? adapter.getPaymentRequirements(mode) : null;
                        refundRequirements = (!payment) ? adapter.getRefundRequirements(mode) : null;
                        TransactionRequirements requirements = payment ? paymentRequirements : refundRequirements;
                        needsLayout |= updateProperty(DESCRIPTION, requirements.getDescription(),
                                                      readOnly || isReversal());
                        needsLayout |= updateProperty(EMAIL, requirements.getEmail(), readOnly);
                    } else {
                        paymentRequirements = null;
                        refundRequirements = null;
                    }
                    refreshRequirements = false;
                } catch (Exception exception) {
                    ErrorHelper.show(exception);
                }
            } else {
                refreshRequirements = false;
            }
        }
        return needsLayout;
    }

    /**
     * Updates a property based on a field requirement and the supplied {@code readOnly} flag.
     *
     * @param name        the property name
     * @param requirement the field requirement
     * @param readOnly    if {@code true}, the property is read-only, else it is editable
     * @return {@code true} if the property was changed, otherwise {@code false}
     */
    private boolean updateProperty(String name, Field requirement, boolean readOnly) {
        MutableProperty property = (MutableProperty) getProperty(name);
        boolean changed;
        if (!readOnly) {
            if (requirement == Field.MANDATORY) {
                changed = property.setEditable(true);
            } else if (requirement == Field.OPTIONAL) {
                changed = property.setEditable(false);
            } else {
                changed = property.setUnsupported();
            }
        } else {
            if (requirement == Field.MANDATORY || requirement == Field.OPTIONAL) {
                changed = property.setViewable();
            } else {
                changed = property.setUnsupported();
            }
        }
        return changed;
    }

    /**
     * Determines if this is a reversal of a payment.
     * <p/>
     * For reversals, the amount and description cannot be changed.
     *
     * @return {@code true} if this is a reversal, oth
     */
    private boolean isReversal() {
        return reversalAmount != null;
    }

    /**
     * Prepares the transaction and submit.
     *
     * @param transaction the transaction
     * @param act         the underlying act
     * @param service     the service adapter
     * @param listener    the listener to notify
     */
    private void prepareAndSubmit(Transaction transaction, FinancialAct act, PaymentProcessorServiceAdapter service,
                                  Runnable listener) {
        ValidationStatus status = service.prepare(transaction);
        boolean isNew = Status.PENDING == transaction.getStatus();
        if (status.getStatus() == ValidationStatus.Status.VALID) {
            submit(transaction, act, isNew, service, listener);
        } else if (status.getStatus() == ValidationStatus.Status.WARNING) {
            ConfirmationDialog.newDialog()
                    .message(status.getMessage())
                    .yesNo()
                    .yes(() -> submit(transaction, act, isNew, service, listener))
                    .show();
        } else {
            Entity paymentProcessor = getPaymentProcessor();
            ErrorHelper.show(paymentProcessor.getName(), status.getMessage());
        }
    }

    /**
     * Checks if a transaction has been updated.
     *
     * @param transaction the transaction
     * @param act         the underlying act
     * @param service     the service adapter
     * @param listener    the listener to notify on success
     */
    private void check(Transaction transaction, FinancialAct act, PaymentProcessorServiceAdapter service,
                       Runnable listener) {
        // transaction has been successfully submitted previously
        if (service.check(transaction)) {
            transactionUpdated(transaction, act, listener);
        } else {
            // nothing changed - transaction still IN_PROGRESS
            listener.run();
        }
    }

    /**
     * Invoked after a transaction has been updated.
     *
     * @param transaction the transaction
     * @param act         the underlying act
     * @param listener    the listener to notify on success
     */
    private void transactionUpdated(Transaction transaction, FinancialAct act, Runnable listener) {
        boolean complete = isComplete(act);
        if (updateTransactionStatus(act, complete)) {
            // change the layout to make properties read-only
            onLayout();
        }

        Entity paymentProcessor = getPaymentProcessor();
        if (transaction.getStatus() == Status.COMPLETED) {
            listener.run();
        } else {
            if (transaction.getStatus() != Status.CANCELLED && transaction.getStatus() != Status.ERROR
                && paymentEditor != null && (paymentEditor.isPosted() || paymentEditor.postOnCompletion())) {
                // transaction is incomplete, so the parent payment cannot be POSTED. Make it IN_PROGRESS so the user
                // doesn't have to.
                paymentEditor.makeSaveable();
            }
            PaymentProcessorStatusDialog dialog = PaymentProcessorStatusDialog.create(paymentProcessor, transaction,
                                                                                      act, getLayoutContext(), false);
            dialog.addWindowPaneListener(new WindowPaneListener() {
                @Override
                public void onClose(WindowPaneEvent event) {
                    listener.run();
                }
            });
            dialog.show();
        }
    }

    /**
     * Determines if a transaction can be cancelled.
     *
     * @param act the transaction
     * @return {@code true} if the transaction can be cancelled, otherwise {@code false}
     */
    private boolean canCancel(FinancialAct act) {
        Status status = Status.valueOf(act.getStatus());
        return status == Status.PENDING || status == Status.IN_PROGRESS || status == Status.SUBMITTED;
    }

    /**
     * Cancels a transaction.
     *
     * @param act the transaction to cancel
     */
    private void cancel(FinancialAct act) {
        Transaction transaction = transactionFactory.getTransaction(act);
        PaymentProcessorServiceAdapter service = getServiceAdapter(transaction.getPaymentProcessor());
        service.cancel(transaction);
    }

    /**
     * Validates that if a refund is being made not linked to a payment, the payment processor supports unmatched
     * refunds.
     *
     * @param validator the validator
     * @return {@code true} if the payment processor is valid, otherwise {@code false}
     */
    private boolean validateUnmatchedRefund(Validator validator) {
        if (getObject().isA(CustomerAccountArchetypes.REFUND_PP) && !supportsUnmatchedRefunds()) {
            if (getBean(getObject()).getSourceRef("reverses") == null) {
                Entity processor = getPaymentProcessor();
                String name = (processor != null) ? processor.getName() : null; // should never be null
                String message = Messages.format("customer.payment.pp.unmatchedRefundsUnsupported", name);
                validator.add(this, message);
            }
        }
        return validator.isValid();
    }

    /**
     * Determines if the payment processor service supports refunds without a corresponding payment.
     *
     * @return {@code true} if it is supported, otherwise {@code false}
     */
    private boolean supportsUnmatchedRefunds() {
        refreshRequirements();
        return refundRequirements != null && refundRequirements.supportsUnmatchedRefunds();
    }

    /**
     * Validates that the amount is the same as that of the act being reversed, if this is a reversal.
     *
     * @param validator the validator
     * @return {@code true} if the amount is valid, otherwise {@code false}
     */
    private boolean validateReversalAmount(Validator validator) {
        if (reversalAmount != null) {
            Property property = getProperty("amount");
            BigDecimal amount = property.getBigDecimal(BigDecimal.ZERO);
            if (amount.compareTo(reversalAmount) != 0) {
                // need to pre-format the amounts as the Messages uses the browser's locale which may have different
                // currency format
                validator.add(property, Messages.format("customer.payment.amountMismatch",
                                                        NumberFormatter.formatCurrency(reversalAmount)));
            }
        }
        return validator.isValid();
    }

    /**
     * Submits a transaction, and notifies the listener on successful completion.
     *
     * @param transaction the transaction
     * @param act         the underlying act
     * @param isNew       if {@code true}, it has not been submitted to the service before
     * @param adapter     the service adapter
     * @param listener    the listener to notify on success
     */
    private void submit(Transaction transaction, FinancialAct act, boolean isNew, PaymentProcessorServiceAdapter adapter,
                        Runnable listener) {
        if (transaction.getStatus() == Status.PENDING) {
            // mark details read-only. The amount cannot change as it would allow the user to change amounts
            // after a submission failure
            transaction.setStatus(Status.IN_PROGRESS);
            onLayout();
        }
        adapter.submit(transaction, isNew);
        transactionUpdated(transaction, act, listener);
    }

    /**
     * Creates a new payment processor transaction act to pass to a payment processor service.
     *
     * @param paymentProcessor the payment processor
     * @param sequence         the relationship sequence
     * @return a new transaction
     */
    private FinancialAct createTransactionAct(Entity paymentProcessor, int sequence) {
        FinancialAct act;
        boolean isPayment = getObject().isA(PAYMENT_PP);
        FinancialAct parent = (FinancialAct) getParent();
        BigDecimal amount = getAmount();
        TransactionMode mode = getTransactionMode();
        String email = getProperty(EMAIL).getString();
        String description = getProperty(DESCRIPTION).getString();
        if (isPayment) {
            act = transactionFactory.createPayment(parent, amount, paymentProcessor, mode, email, description);
        } else {
            act = transactionFactory.createRefund(parent, amount, paymentProcessor, mode, email, description);
        }
        addTransaction(act, sequence);
        return act;
    }

    /**
     * Returns a service adapter for a payment processor.
     *
     * @param config the payment processor configuration
     * @return the corresponding service
     * @throws PaymentProcessorException if the service is unavailable
     */
    private PaymentProcessorServiceAdapter getServiceAdapter(Entity config) {
        return new PaymentProcessorServiceAdapter(getPaymentProcessorService(config));
    }

}