/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.component.business.domain.im.datatypes.quantity.Money;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.im.act.ActHelper;
import org.openvpms.web.component.im.delete.Deletable;
import org.openvpms.web.component.im.edit.act.ActRelationshipCollectionEditor;
import org.openvpms.web.component.im.edit.act.ParticipationEditor;
import org.openvpms.web.component.im.edit.payment.PaymentEditor;
import org.openvpms.web.component.im.edit.payment.PaymentItemEditor;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.IMObjectCreationListener;
import org.openvpms.web.component.property.CollectionProperty;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.SimpleProperty;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.resource.i18n.format.NumberFormatter;
import org.openvpms.web.system.ServiceHelper;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;

/**
 * An editor for {@link Act}s which have an archetype of
 * <em>act.customerAccountPayment</em> or <em>act.customerAccountRefund</em>.
 *
 * @author Tim Anderson
 */
public abstract class AbstractCustomerPaymentEditor extends PaymentEditor {

    /**
     * The payment status.
     */
    private final PaymentStatus status;

    /**
     * The amount of the invoice that this payment relates to.
     */
    private final SimpleProperty invoiceAmount;

    /**
     * The customer account rules.
     */
    private final CustomerAccountRules rules;

    /**
     * The expected amount, if this is a reversal.
     */
    private final BigDecimal reversalAmount;

    /**
     * Determines the expected payment amount. If {@code null}, there
     * is no limit on the payment amount. If non-null, validation will fail
     * if the act total is not that specified.
     */
    private BigDecimal expectedAmount;

    /**
     * The last till used to successfully make an EFT payment/refund.
     * <p/>
     * Once an EFT payment/refund has been made, the till cannot be changed.
     */
    private Entity eftTill;

    /**
     * Determines if a default item should be added if no items are present.
     */
    private boolean addDefaultItem;

    /**
     * Constructs an {@link AbstractCustomerPaymentEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent object. May be {@code null}
     * @param context the layout context
     */
    public AbstractCustomerPaymentEditor(FinancialAct act, IMObject parent, LayoutContext context) {
        this(act, parent, context, act.isNew());
    }

    /**
     * Constructs an {@link AbstractCustomerPaymentEditor}.
     *
     * @param act            the act to edit
     * @param parent         the parent object. May be {@code null}
     * @param context        the layout context
     * @param addDefaultItem if {@code true} add a default item if the act has none
     */
    public AbstractCustomerPaymentEditor(FinancialAct act, IMObject parent, LayoutContext context,
                                         boolean addDefaultItem) {
        super(act, parent, context);
        status = new PaymentStatus(getProperty("status"), act);
        invoiceAmount = createProperty("invoiceAmount", "customer.payment.currentInvoice");
        rules = ServiceHelper.getBean(CustomerAccountRules.class);
        this.addDefaultItem = addDefaultItem;

        FinancialAct reversal = getBean(act).getSource("reverses", FinancialAct.class);
        reversalAmount = (reversal != null) ? reversal.getTotal() : null;

        initParticipant("customer", context.getContext().getCustomer());
        initParticipant("location", context.getContext().getLocation());

        if (!status.isReadOnly()) {
            addEditor(status.getEditor());
        }

        getItems().setCreationListener(new IMObjectCreationListener() {
            public void created(IMObject object) {
                onCreated((FinancialAct) object);
            }
        });

        if (!act.isNew()) {
            Entity till = getTill();
            if (till != null) {
                for (EFTPaymentItemEditor itemEditor : getEFTItemEditors()) {
                    if (!itemEditor.canDelete()) {
                        // the till can't be changed if a transaction has been approved, or one is in progress
                        eftTill = till;
                    }
                }
            }
        }
    }

    /**
     * Returns the customer.
     *
     * @return the customer. May be {@code null}
     */
    public Party getCustomer() {
        return (Party) getParticipant("customer");
    }

    /**
     * Determines if the current status is POSTED.
     *
     * @return {@code true} if the current status is POSTED.
     */
    public boolean isPosted() {
        return getStatus().equals(ActStatus.POSTED);
    }

    /**
     * If the current status is POSTED, sets it to IN_PROGRESS.
     */
    public void makeSaveable() {
        status.makeSaveable();
    }

    /**
     * If the current status is POSTED, sets it to IN_PROGRESS, and indicate that it should be set to POSTED on
     * completion of editing. This enables the payment to be saved without affecting the balance (once POSTED,
     * totals cannot change).
     * <p/>
     * The view will still indicate POSTED.
     */
    public void makeSaveableAndPostOnCompletion() {
        status.makeSaveableAndPostOnCompletion();
    }

    /**
     * Sets the invoice amount.
     *
     * @param amount the invoice amount
     */
    public void setInvoiceAmount(BigDecimal amount) {
        invoiceAmount.setValue(amount);
    }

    /**
     * Returns the invoice amount.
     *
     * @return the invoice amount
     */
    public BigDecimal getInvoiceAmount() {
        return invoiceAmount.getBigDecimal(BigDecimal.ZERO);
    }

    /**
     * Determines the expected amount of the payment. If {@code null}, there
     * is no limit on the payment amount. If non-null, validation will fail
     * if the act total is not that specified.
     *
     * @param amount the expected payment amount. May be {@code null}
     */
    public void setExpectedAmount(BigDecimal amount) {
        expectedAmount = amount;
    }

    /**
     * Returns the expected amount of the payment.
     *
     * @return the expected amount of the payment. May be {@code null}
     */
    public BigDecimal getExpectedAmount() {
        return expectedAmount;
    }

    /**
     * Sets the practice location.
     *
     * @param location the practice location
     */
    public void setLocation(Party location) {
        setParticipant("location", location);
    }

    /**
     * Returns the practice location.
     *
     * @return the practice location. May be {@code null}
     */
    public Party getLocation() {
        return (Party) getParticipant("location");
    }

    /**
     * Sets the till.
     *
     * @param till the till. May be {@code null}
     */
    public void setTill(Entity till) {
        setParticipant("till", till);
        if (!getView().hasComponent()) { // if there is no editor component yet, need to trigger update manually
            onTillChanged(); //
        }
    }

    /**
     * Returns the selected till.
     *
     * @return the till, or {@code null} if none has been selected
     */
    public Entity getTill() {
        return getParticipant("till");
    }

    /**
     * Determines if there are any 3rd-party transactions that need to be performed or completed.
     * <p/>
     * This only applies to acts that aren't saved POSTED.
     *
     * @return {@code true} if there are items requiring 3rd party transactions to be performed, otherwise {@code false}
     */
    public boolean requiresTransaction() {
        return getRequiresTransactionEditor() != null;
    }

    /**
     * Returns the first editor that requires a 3rd-party transaction to be performed or completed.
     * <p/>
     * This only applies to acts that aren't saved POSTED.
     *
     * @return the first editor, or {@code null} if none require transactions to be performed
     */
    public TransactionPaymentItemEditor getRequiresTransactionEditor() {
        TransactionPaymentItemEditor editor = null;
        if (!status.isReadOnly()) {
            editor = getTransactionPaymentItemEditors().stream()
                    .filter(TransactionPaymentItemEditor::requiresTransaction)
                    .findFirst()
                    .orElse(null);
        }
        return editor;
    }

    /**
     * Processes incomplete transactions.
     * <p/>
     * This should be invoked when {@link #requiresTransaction()} returns {@code true}.
     * <p/>
     * Note that while the listener will be notified on success, transactions may not be complete.
     *
     * @param listener the listener to notify on success
     */
    public void processTransactions(Runnable listener) {
        if (!status.isReadOnly() && isValid()) {
            List<TransactionPaymentItemEditor> editors = getTransactionPaymentItemEditors();
            processTransactions(editors.iterator(), listener);
        }
    }

    /**
     * Determines if the act should be posted on completion of the edit.
     *
     * @return {@code true} if the act should be posted on completion of the edit
     */
    public boolean postOnCompletion() {
        return status.postOnCompletion();
    }

    /**
     * Determines if the status can be changed.
     *
     * @return {@code true} if the status can be changed
     */
    public boolean canChangeStatus() {
        return !status.isReadOnly();
    }

    /**
     * Determines if the payment can be deleted.
     *
     * @return {@code true} if the payment can be deleted, otherwise {@code false}
     */
    public boolean canDelete() {
        boolean result = false;
        if (!getObject().isNew() && canChangeStatus()) {
            result = true;
            for (PaymentItemEditor editor : getPaymentItemEditors()) {
                if (!editor.canDelete()) {
                    result = false;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Determines if the payment is empty, or only has EFT or payment processor items.
     *
     * @return {@code true} if the payment only has EFT or payment processor items
     */
    public boolean isEmptyOrOnlyHasTransactionItems() {
        boolean result;
        List<Act> items = getItems().getCurrentActs();
        if (items.isEmpty()) {
            result = true;
        } else {
            result = items.stream().allMatch(act -> act.isA(PAYMENT_EFT, REFUND_EFT, PAYMENT_PP, REFUND_PP));
        }
        return result;
    }

    /**
     * Determines if a default item should be added if the charge doesn't have one.
     * <p/>
     * This only applies prior to the creation of the component. After that, it is ignored.
     *
     * @param addDefaultItem if {@code true} add a default item if the charge has none
     */
    public void setAddDefaultItem(boolean addDefaultItem) {
        this.addDefaultItem = addDefaultItem;
    }

    /**
     * Creates a collection editor for the items collection.
     *
     * @param act   the act
     * @param items the items collection
     * @return a new collection editor
     */
    @Override
    protected ActRelationshipCollectionEditor createItemsEditor(Act act, CollectionProperty items) {
        PaymentItemRelationshipCollectionEditor editor
                = new PaymentItemRelationshipCollectionEditor(items, act, getLayoutContext());
        editor.setParent(this);
        return editor;
    }

    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        return new CustomerPaymentLayoutStrategy(getItems(), getPaymentStatus());
    }

    /**
     * Invoked when layout has completed.
     * <p>
     * This can be used to perform processing that requires all editors to be created.
     */
    @Override
    protected void onLayoutCompleted() {
        super.onLayoutCompleted();
        initItems();
        ParticipationEditor<Entity> till = getParticipationEditor("till", true);
        if (till != null) {
            till.addModifiableListener(modifiable -> onTillChanged());
        }
    }

    /**
     * Returns the payment status.
     *
     * @return the payment status
     */
    protected PaymentStatus getPaymentStatus() {
        return status;
    }

    /**
     * Validates the object.
     * <p/>
     * This extends validation by ensuring that the payment amount matches the expected amount, if present.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        boolean requiresTransaction = requiresTransaction();
        if (requiresTransaction && status.isPosted()) {
            // Cannot save the transaction as POSTED when EFT or payment processor transactions are incomplete
            status.makeSaveableAndPostOnCompletion();
        }
        boolean valid = super.doValidation(validator) && validateExpectedAmount(validator)
                        && validateTill(validator) && validateReversal(validator);
        if (valid) {
            if (requiresTransaction && ActStatus.POSTED.equals(getStatus())) {
                // Shouldn't occur. Status has been overridden
                validator.add(this, "Cannot finalise payment until EFT transactions are complete");
            }
        }
        return validator.isValid();
    }

    /**
     * Verifies any conditions that must be met prior to performing deletion.
     *
     * @throws IllegalStateException if deletion is not allowed
     */
    @Override
    protected void checkDeletePreconditions() {
        super.checkDeletePreconditions();
        for (PaymentItemEditor editor : getPaymentItemEditors()) {
            if (!editor.canDelete()) {
                if (!(editor instanceof TransactionPaymentItemEditor)
                    || !((TransactionPaymentItemEditor) editor).canCancelTransaction()) {
                    throw new IllegalStateException("Cannot delete payment item: " + editor.getDisplayName());
                }
            }
        }
    }

    /**
     * Prepares the object for deletion.
     *
     * @throws OpenVPMSException if the object cannot be prepared for deletion
     */
    @Override
    protected void prepareDelete() {
        super.prepareDelete();
        for (PaymentItemEditor editor : getPaymentItemEditors()) {
            if (editor instanceof TransactionPaymentItemEditor) {
                TransactionPaymentItemEditor transactionEditor = (TransactionPaymentItemEditor) editor;
                Deletable deletable = transactionEditor.getDeletable();
                if (!deletable.canDelete()) {
                    if (transactionEditor.canCancelTransaction()) {
                        transactionEditor.cancelTransaction();
                        Deletable postCancel = transactionEditor.getDeletable();
                        if (!postCancel.canDelete()) {
                            throw new PaymentException(postCancel.getReason());
                        }
                    } else {
                        throw new PaymentException(deletable.getReason());
                    }
                }
            }
        }
    }

    /**
     * Deletes the object.
     *
     * @throws OpenVPMSException     if the delete fails
     * @throws IllegalStateException if the act is POSTED or has payment items that cannot be deleted
     */
    @Override
    protected void doDelete() {
        List<PaymentItemEditor> editors = getPaymentItemEditors();

        for (PaymentItemEditor editor : editors) {
            if (!editor.canDelete()) {
                throw new IllegalStateException("Cannot delete payment item: " + editor.getDisplayName());
            }
        }
        super.doDelete();
    }

    /**
     * Returns the EFT item editors.
     *
     * @return the EFT item editors
     */
    protected List<EFTPaymentItemEditor> getEFTItemEditors() {
        return getTransactionPaymentItemEditors().stream()
                .filter(editor -> editor instanceof EFTPaymentItemEditor)
                .map(editor -> (EFTPaymentItemEditor) editor)
                .collect(Collectors.toList());
    }

    /**
     * Returns the item editors that support 3rd-party transactions.
     *
     * @return the transaction payment item editors
     */
    protected List<TransactionPaymentItemEditor> getTransactionPaymentItemEditors() {
        return getPaymentItemEditors().stream()
                .filter(editor -> editor instanceof TransactionPaymentItemEditor)
                .map(editor -> (TransactionPaymentItemEditor) editor)
                .collect(Collectors.toList());
    }


    /**
     * Returns the payment item editors.
     *
     * @return the payment item editors
     */
    protected List<PaymentItemEditor> getPaymentItemEditors() {
        ActRelationshipCollectionEditor items = getItems();
        return items.getCurrentActs().stream()
                .map(items::getEditor)
                .filter(editor -> editor instanceof PaymentItemEditor)
                .map(editor -> (PaymentItemEditor) editor)
                .collect(Collectors.toList());
    }

    /**
     * Invoked when a child act is created. This sets the total to the:
     * <ul>
     * <li>outstanding balance +/- the running total, if there is no expected amount; or</li>
     * <li>expected amount - the running total</li>
     * </ul>
     *
     * @param act the act
     */
    protected void onCreated(FinancialAct act) {
        Party customer = (Party) getParticipant("customer");
        if (customer != null) {
            BigDecimal runningTotal = getRunningTotal();
            BigDecimal balance;
            if (expectedAmount == null) {
                // default the amount to the outstanding balance +/- the running total.
                boolean payment = act.isA("act.customerAccountPayment*");
                balance = rules.getBalance(customer, runningTotal, payment);
                act.setTotal(new Money(balance));
            } else {
                // default the amount to the expected amount - the running total.
                balance = expectedAmount.subtract(runningTotal);
                if (balance.signum() >= 0) {
                    act.setTotal(new Money(balance));
                }
            }
            getItems().setModified(act, true);
        }
    }

    /**
     * Returns the invoice amount property.
     *
     * @return the property
     */
    protected Property getInvoiceAmountProperty() {
        return invoiceAmount;
    }

    /**
     * Helper to create a decimal property.
     *
     * @param name the property name
     * @param key  the resource bundle key
     * @return a new property
     */
    protected SimpleProperty createProperty(String name, String key) {
        SimpleProperty property = new SimpleProperty(name, BigDecimal.class);
        property.setDisplayName(Messages.get(key));
        property.setReadOnly(true);
        return property;
    }

    /**
     * Validates that the amount is the same as the expected amount, if present.
     *
     * @param validator the validator
     * @return {@code true} if valid, otherwise {@code false}
     */
    private boolean validateExpectedAmount(Validator validator) {
        if (expectedAmount != null) {
            validateAmount(expectedAmount, validator);
        }
        return validator.isValid();
    }

    /**
     * Validates that the amount is the same as that supplied.
     *
     * @param expected  the expected amount
     * @param validator the validator
     */
    private void validateAmount(BigDecimal expected, Validator validator) {
        Property property = getProperty("amount");
        BigDecimal amount = property.getBigDecimal(BigDecimal.ZERO);
        if (amount.compareTo(expected) != 0) {
            // need to pre-format the amounts as the Messages uses the browser's locale which may have different
            // currency format
            validator.add(property, Messages.format("customer.payment.amountMismatch",
                                                    NumberFormatter.formatCurrency(expected)));
        }
    }

    /**
     * Validates that the last till used for EFT is the same as the main till, if present.
     *
     * @param validator the validator
     * @return {@code true} if valid, otherwise {@code false}
     */
    private boolean validateTill(Validator validator) {
        if (eftTill != null && !eftTill.equals(getTill())) {
            validator.add(this, Messages.get("customer.payment.eft.cannotchangetill"));
        }
        return validator.isValid();
    }

    /**
     * Ensures that if this is a reversal, the amount corresponds to that of the transaction being reversed.
     *
     * @param validator the validator
     * @return {@code true} if valid, otherwise {@code false}
     */
    private boolean validateReversal(Validator validator) {
        if (reversalAmount != null) {
            validateAmount(reversalAmount, validator);
        }
        return validator.isValid();
    }

    /**
     * Recursively processes transactions.
     * <p/>
     * Note that while the listener will be notified on success, transactions may not be complete.
     *
     * @param iterator the iterator over payment item editors that manage transactions
     * @param listener the listener to notify on success
     */
    private void processTransactions(Iterator<TransactionPaymentItemEditor> iterator, Runnable listener) {
        if (iterator.hasNext()) {
            TransactionPaymentItemEditor editor = iterator.next();
            editor.performTransaction(() -> processTransactions(iterator, listener));
        } else {
            listener.run();
        }
    }

    /**
     * Adds a default item if the object is new and there are no items present.
     */
    private void initItems() {
        if (addDefaultItem) {
            ActRelationshipCollectionEditor items = getItems();
            CollectionProperty property = items.getCollection();
            if (property.getValues().isEmpty()) {
                addItem();
            }
        }
    }

    /**
     * Invoked when the till changes. Propagates it to the EFT item editors.
     */
    private void onTillChanged() {
        Entity till = getTill();
        if (eftTill == null || !eftTill.equals(till)) {
            for (EFTPaymentItemEditor itemEditor : getEFTItemEditors()) {
                itemEditor.setTill(till);
            }
        }
    }

    /**
     * Returns the running total. This is the current total of the act
     * minus any committed child acts which are already included in the balance.
     *
     * @return the running total
     */
    private BigDecimal getRunningTotal() {
        FinancialAct act = getObject();
        BigDecimal total = act.getTotal();
        BigDecimal committed = ActHelper.sum(act, "amount");
        return total.subtract(committed);
    }
}
