/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.charge;

import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.prefs.Preferences;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.query.ActQuery;
import org.openvpms.web.component.im.query.ActStatuses;
import org.openvpms.web.component.im.query.Browser;
import org.openvpms.web.component.im.query.BrowserFactory;
import org.openvpms.web.component.im.query.Query;
import org.openvpms.web.component.im.table.IMObjectTableModel;
import org.openvpms.web.component.workspace.CRUDWindow;
import org.openvpms.web.workspace.customer.CustomerFinancialWorkspace;
import org.openvpms.web.workspace.customer.account.CustomerAccountActTableModel;
import org.openvpms.web.workspace.customer.account.CustomerAccountQuery;


/**
 * Customer charges workspace.
 *
 * @author Tim Anderson
 */
public class ChargeWorkspace extends CustomerFinancialWorkspace<FinancialAct> {

    /**
     * The act statuses to query, excluding POSTED.
     */
    private static final ActStatuses STATUSES = new ActStatuses(
            "act.customerAccountChargesInvoice", FinancialActStatus.POSTED);

    /**
     * The customer archetype short names.
     */
    private static final String[] CUSTOMER_SHORT_NAMES = {"party.customer*", "party.organisationOTC"};

    /**
     * Constructs a {@link ChargeWorkspace}.
     *
     * @param context     the context
     * @param preferences the user preferences
     */
    public ChargeWorkspace(Context context, Preferences preferences) {
        super("customer.charge", context, preferences);
        setArchetypes(Party.class, CUSTOMER_SHORT_NAMES);
        setChildArchetypes(FinancialAct.class, "act.customerAccountCharges*");
    }

    /**
     * Creates a new CRUD window for viewing and editing acts.
     *
     * @return a new CRUD window
     */
    protected CRUDWindow<FinancialAct> createCRUDWindow() {
        return new ChargeCRUDWindow(getChildArchetypes(), getContext(), getHelpContext());
    }

    /**
     * Creates a new query.
     *
     * @return a new query
     */
    protected ActQuery<FinancialAct> createQuery() {
        return new CustomerAccountQuery<>(getObject(), getChildArchetypes().getShortNames(),
                                          STATUSES, new DefaultLayoutContext(getContext(), getHelpContext()));
    }

    /**
     * Invoked when the object has been saved.
     *
     * @param object the object
     * @param isNew  determines if the object is a new instance
     */
    @Override
    protected void onSaved(FinancialAct object, boolean isNew) {
        super.onSaved(object, isNew);
        if (FinancialActStatus.POSTED.equals(object.getStatus())) {
            onBrowserSelected(null);
        }
    }

    /**
     * Creates a new browser to query and display acts.
     *
     * @param query the query
     * @return a new browser
     */
    @Override
    protected Browser<FinancialAct> createBrowser(Query<FinancialAct> query) {
        IMObjectTableModel<FinancialAct> model = new CustomerAccountActTableModel<>(true, false);
        return BrowserFactory.create(query, null, model, new DefaultLayoutContext(getContext(), getHelpContext()));
    }

}
