/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace;

import echopointng.ContentPaneEx;
import nextapp.echo2.app.SplitPane;
import org.openvpms.archetype.rules.doc.LogoService;
import org.openvpms.archetype.rules.practice.PracticeRules;
import org.openvpms.archetype.rules.user.UserRules;
import org.openvpms.web.component.app.GlobalContext;
import org.openvpms.web.component.prefs.UserPreferences;
import org.openvpms.web.component.workspace.WorkspacesFactory;
import org.openvpms.web.echo.factory.SplitPaneFactory;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.workflow.messaging.messages.MessageMonitor;


/**
 * Content pane that displays the {@link TitlePane} and {@link MainPane}.
 *
 * @author Tim Anderson
 */
public class ApplicationContentPane extends ContentPaneEx {

    /**
     * The context.
     */
    private final GlobalContext context;

    /**
     * The workspaces factory.
     */
    private final WorkspacesFactory factory;

    /**
     * The user preferences
     */
    private final UserPreferences preferences;

    /**
     * The layout pane style name.
     */
    private static final String LAYOUT_STYLE = "ApplicationContentPane.Layout";


    /**
     * Constructs an {@link ApplicationContentPane}.
     *
     * @param context     the context
     * @param factory     the workspaces factory
     * @param preferences the user preferences
     */
    public ApplicationContentPane(GlobalContext context, WorkspacesFactory factory, UserPreferences preferences) {
        this.context = context;
        this.factory = factory;
        this.preferences = preferences;
    }

    /**
     * @see nextapp.echo2.app.Component#init()
     */
    public void init() {
        super.init();
        doLayout();
    }

    /**
     * Lays out the component.
     */
    protected void doLayout() {
        SplitPane layout = SplitPaneFactory.create(SplitPane.ORIENTATION_VERTICAL, LAYOUT_STYLE);
        layout.add(createTitlePane());
        layout.add(createMainPane());
        add(layout);
    }

    /**
     * Creates a new title pane.
     *
     * @return a new title pane
     */
    protected TitlePane createTitlePane() {
        PracticeRules practiceRules = ServiceHelper.getBean(PracticeRules.class);
        UserRules userRules = ServiceHelper.getBean(UserRules.class);
        LogoService logoService = ServiceHelper.getBean(LogoService.class);
        return new TitlePane(practiceRules, userRules, context, preferences, logoService);
    }

    /**
     * Creates a new main pane.
     *
     * @return a new main pane
     */
    protected MainPane createMainPane() {
        MessageMonitor messageMonitor = ServiceHelper.getBean(MessageMonitor.class);
        return new MainPane(messageMonitor, context, factory, preferences);
    }

    /**
     * Returns the context.
     *
     * @return the context
     */
    protected GlobalContext getContext() {
        return context;
    }

    /**
     * Returns the workspaces factory.
     *
     * @return the workspaces factory
     */
    protected WorkspacesFactory getWorkspacesFactory() {
        return factory;
    }

    /**
     * Returns the user preferences.
     *
     * @return the user preferences
     */
    protected UserPreferences getPreferences() {
        return preferences;
    }

}
