/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.workflow.otc;

import nextapp.echo2.app.Component;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.system.common.util.Variables;
import org.openvpms.web.component.im.layout.ComponentGrid;
import org.openvpms.web.component.im.layout.ComponentSet;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.PropertySet;
import org.openvpms.web.component.property.PropertySetBuilder;
import org.openvpms.web.workspace.customer.payment.AbstractCustomerPaymentEditor;
import org.openvpms.web.workspace.customer.payment.CustomerPaymentLayoutStrategy;

import java.math.BigDecimal;
import java.util.List;

/**
 * Editor for over-the-counter payments.
 * <p/>
 * This ensures that the payment total is the same as the charge.
 * <p/>
 * It also suppresses changing of the act status, in case the workflow is cancelled and the act needs to be deleted.
 *
 * @author Tim Anderson
 */
class OTCPaymentEditor extends AbstractCustomerPaymentEditor {

    /**
     * Constructs an {@link OTCPaymentEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent object. May be {@code null}
     * @param context the layout context
     * @param charge  the charge amount
     */
    public OTCPaymentEditor(FinancialAct act, IMObject parent, LayoutContext context, BigDecimal charge) {
        super(act, parent, context);
        setInvoiceAmount(charge);
        setExpectedAmount(charge);
    }

    /**
     * Creates the property set.
     * <p/>
     * This implementation modifies the <em>items</em> collection, to remove the <em>act.customerAccountPaymentPP</em>
     * and <em>act.customerAccountRefundPP</em> archetypes, as these aren't applicable to OTC.
     *
     * @param object    the object being edited
     * @param archetype the object archetype
     * @param variables the variables for macro expansion. May be {@code null}
     * @return the property set
     */
    @Override
    protected PropertySet createPropertySet(IMObject object, ArchetypeDescriptor archetype, Variables variables) {
        return new PropertySetBuilder(object, archetype, variables)
                .removeArchetypesFromRange("items", CustomerAccountArchetypes.PAYMENT_PP,
                                           CustomerAccountArchetypes.REFUND_PP)
                .build();
    }


    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        return new LayoutStrategy();
    }

    private class LayoutStrategy extends CustomerPaymentLayoutStrategy {

        /**
         * Constructs a {@link LayoutStrategy}.
         */
        public LayoutStrategy() {
            super(getItems(), getPaymentStatus());
            // don't allow editing of the status
            getArchetypeNodes().exclude("status");
        }

        /**
         * Lays out child components in a grid.
         *
         * @param object     the object to lay out
         * @param parent     the parent object. May be {@code null}
         * @param properties the properties
         * @param container  the container to use
         * @param context    the layout context
         */
        @Override
        protected void doSimpleLayout(IMObject object, IMObject parent, List<Property> properties,
                                      Component container, LayoutContext context) {
            ComponentSet set = createComponentSet(object, properties, context);
            ComponentGrid grid = new ComponentGrid();
            grid.add(createComponent(getInvoiceAmountProperty(), object, context));
            grid.add(set);
            addAuditInfo(object, grid, context);
            doGridLayout(grid, container);
        }
    }
}
