/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.reporting.statement;

import org.openvpms.archetype.component.processor.Processor;
import org.openvpms.archetype.rules.customer.CustomerArchetypes;
import org.openvpms.archetype.rules.finance.statement.StatementProcessorException;
import org.openvpms.archetype.rules.finance.statement.StatementService;
import org.openvpms.component.business.service.archetype.ArchetypeServiceHelper;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.rule.IArchetypeRuleService;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IterableIMObjectQuery;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

import java.util.Date;


/**
 * End-of-period generator.
 *
 * @author Tim Anderson
 */
class EndOfPeriodGenerator extends AbstractStatementGenerator {

    /**
     * The statement processor progress bar.
     */
    private final StatementProgressBarProcessor progressBarProcessor;


    /**
     * Constructs an {@link EndOfPeriodGenerator}.
     *
     * @param date                 the statement date
     * @param postCompletedCharges if {@code true} post completed charge acts
     * @param context              the context
     * @param help                 the help context
     */
    public EndOfPeriodGenerator(Date date, boolean postCompletedCharges, Context context, HelpContext help) {
        super(Messages.get("reporting.statements.eop.title"),
              Messages.get("reporting.statements.eop.cancel.title"),
              Messages.get("reporting.statements.eop.cancel.message"),
              Messages.get("reporting.statements.eop.retry.title"), help);
        Party practice = context.getPractice();
        if (practice == null) {
            throw new StatementProcessorException(StatementProcessorException.ErrorCode.InvalidConfiguration,
                                                  "Context has no practice");
        }

        StatementService statementService = ServiceHelper.getBean(StatementService.class);
        ArchetypeQuery query = new ArchetypeQuery(CustomerArchetypes.PERSON, false, false);
        int size = countCustomers(query);
        query.add(Constraints.sort("id"));
        query.setMaxResults(1000);

        IArchetypeRuleService service = ServiceHelper.getArchetypeService();
        IterableIMObjectQuery<Party> customers = new IterableIMObjectQuery<>(service, query);

        Processor<Party> processor = object -> statementService.endPeriod(object, date, postCompletedCharges);
        progressBarProcessor = new StatementProgressBarProcessor(processor, customers, size);
    }

    /**
     * Returns the processor.
     *
     * @return the processor
     */
    protected StatementProgressBarProcessor getProcessor() {
        return progressBarProcessor;
    }

    /**
     * Counts the available customers matching a query.
     *
     * @param query the query
     * @return the no. of customers
     */
    private int countCustomers(ArchetypeQuery query) {
        query.setMaxResults(0);
        IArchetypeService service = ArchetypeServiceHelper.getArchetypeService();
        query.setCountResults(true);
        int size = service.get(query).getTotalResults();
        query.setCountResults(false);
        return size;
    }
}
