/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.product.stock;

import org.openvpms.archetype.rules.stock.StockRules;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.web.component.im.edit.act.ActItemEditor;
import org.openvpms.web.component.im.layout.ComponentSet;
import org.openvpms.web.component.im.layout.IMObjectLayoutStrategy;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.view.ComponentState;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.component.property.SimpleProperty;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

import java.math.BigDecimal;
import java.util.List;


/**
 * An editor for <em>act.stockTransferItem</em> acts.
 *
 * @author Tim Anderson
 */
public class StockTransferItemEditor extends ActItemEditor {

    /**
     * Displays the current stock at the 'from' location.
     */
    private final SimpleProperty fromQuantity;

    /**
     * Displays the current stock at the 'to' location.
     */
    private final SimpleProperty toQuantity;

    /**
     * The stock rules.
     */
    private final StockRules rules;

    /**
     * The transfer-from location.
     */
    private Party transferFrom;

    /**
     * The transfer-to location
     */
    private Party transferTo;


    /**
     * Constructs a {@link StockTransferItemEditor}.
     *
     * @param act     the act to edit
     * @param parent  the parent act
     * @param context the layout context
     */
    public StockTransferItemEditor(Act act, Act parent, LayoutContext context) {
        super(act, parent, context);
        rules = ServiceHelper.getBean(StockRules.class);
        fromQuantity = new SimpleProperty("fromQuantity", BigDecimal.class);
        fromQuantity.setDisplayName(Messages.get("product.stock.fromQuantity"));
        fromQuantity.setValue(BigDecimal.ZERO);
        fromQuantity.setReadOnly(true);

        toQuantity = new SimpleProperty("toQuantity", BigDecimal.class);
        toQuantity.setDisplayName(Messages.get("product.stock.toQuantity"));
        toQuantity.setValue(BigDecimal.ZERO);
        toQuantity.setReadOnly(true);

        if (parent != null) {
            IMObjectBean bean = getBean(parent);
            setTransferFrom((Party) getObject(bean.getTargetRef("stockLocation")));
            setTransferTo((Party) getObject(bean.getTargetRef("to")));
        }
    }

    /**
     * Sets the 'transfer from' stock location.
     *
     * @param location the stock location to transfer from
     */
    public void setTransferFrom(Party location) {
        transferFrom = location;
        updateFromQuantity(getProduct());
    }

    /**
     * Sets the 'transfer to' stock location.
     *
     * @param location the stock location to transfer to
     */
    public void setTransferTo(Party location) {
        transferTo = location;
        updateToQuantity(getProduct());
    }

    /**
     * Invoked when the product is changed.
     *
     * @param product the product. May be {@code null}
     */
    @Override
    protected void productModified(Product product) {
        updateFromQuantity(product);
        updateToQuantity(product);
        notifyProductListener(product);
    }

    /**
     * Creates the layout strategy.
     *
     * @return a new layout strategy
     */
    @Override
    protected IMObjectLayoutStrategy createLayoutStrategy() {
        return new LayoutStrategy() {
            @Override
            protected ComponentSet createComponentSet(
                    IMObject object, List<Property> properties,
                    LayoutContext context) {
                ComponentSet set = super.createComponentSet(object, properties,
                                                            context);
                ComponentState from = createComponent(fromQuantity, object,
                                                      context);
                ComponentState to = createComponent(toQuantity, object,
                                                    context);
                set.add(from);
                set.add(to);
                return set;
            }
        };
    }

    /**
     * Updates the transfer-from stock quantity.
     *
     * @param product the product. May be {@code null}
     */
    private void updateFromQuantity(Product product) {
        fromQuantity.setValue(getStock(product, transferFrom));
    }

    /**
     * Updates the transfer-to stock quantity.
     *
     * @param product the product. May be {@code null}
     */
    private void updateToQuantity(Product product) {
        toQuantity.setValue(getStock(product, transferTo));
    }

    /**
     * Returns the stock at the specified stock location.
     *
     * @param product       the product
     * @param stockLocation the stock location
     * @return the stock quantity
     */
    private BigDecimal getStock(Product product, Party stockLocation) {
        BigDecimal result = BigDecimal.ZERO;
        try {
            if (product != null && stockLocation != null) {
                result = rules.getStock(product, stockLocation);
            }
        } catch (OpenVPMSException error) {
            ErrorHelper.show(error);
        }
        return result;
    }
}
