/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.patient.problem;

import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.web.workspace.patient.history.AbstractPatientHistoryFilter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

/**
 * Filters patient problems.
 * <p>
 * This enables specific problems items to by included by archetype.
 *
 * @author Tim Anderson
 */
public class ProblemFilter extends AbstractPatientHistoryFilter {

    /**
     * The search criteria.
     */
    private final Predicate<org.openvpms.component.model.act.Act> search;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * Constructs an {@link ProblemFilter}.
     *
     * @param archetypes the act archetypes
     * @param search     the search criteria. May be {@code null}
     * @param ascending  if {@code true} sort items on ascending timestamp; otherwise sort on descending timestamp
     * @param service    the archetype service
     */
    public ProblemFilter(String[] archetypes, Predicate<org.openvpms.component.model.act.Act> search,
                         boolean ascending, ArchetypeService service) {
        super(PatientArchetypes.CLINICAL_PROBLEM, archetypes, Collections.emptyList());
        this.search = search;
        this.service = service;
        setSortItemsAscending(ascending);
    }

    /**
     * Returns a comparator to sort the children of an act.
     *
     * @param act the parent act
     * @return the comparator to sort the act's children
     */
    @Override
    public Comparator<Act> getComparator(Act act) {
        if (act.isA(PatientArchetypes.PATIENT_MEDICATION, PatientArchetypes.CLINICAL_NOTE)) {
            return super.getComparator(true);
        }
        return super.getComparator(act);
    }

    /**
     * Filters child acts.
     *
     * @param parent   the parent act
     * @param children the child acts
     * @param acts     the set of visited acts, keyed on reference
     * @return the filtered acts
     */
    @Override
    protected List<Act> filter(Act parent, List<Act> children, Map<Reference, Act> acts) {
        List<Act> result;
        if (search == null) {
            result = children;
        } else {
            result = new ArrayList<>();
            for (Act act : children) {
                // events always match, so that the child acts can be searched.
                if (act.isA(PatientArchetypes.CLINICAL_EVENT) || search.test(act)) {
                    result.add(act);
                }
            }
        }
        return result;
    }

    /**
     * Returns the immediate children of an act.
     *
     * @param act  the parent act
     * @param acts a cache of the visited acts, keyed on reference
     * @return the immediate children of {@code act}
     */
    @Override
    protected Set<Act> getChildren(Act act, Map<Reference, Act> acts) {
        Set<Act> children = super.getChildren(act, acts);
        if (act.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
            IMObjectBean bean = service.getBean(act);
            Set<Act> events = getActs(bean.getSourceRefs("events"), acts);
            children.addAll(events);
        }
        return children;
    }

}
