/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.patient.history;

import nextapp.echo2.app.Column;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.Label;
import nextapp.echo2.app.Row;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.NodeSelectConstraint;
import org.openvpms.component.system.common.query.ObjectSet;
import org.openvpms.component.system.common.query.ObjectSetQueryIterator;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.LookupNameHelper;
import org.openvpms.web.echo.factory.ColumnFactory;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.factory.RowFactory;
import org.openvpms.web.echo.style.Styles;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.resource.i18n.format.NumberFormatter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static org.openvpms.component.system.common.query.Constraints.shortName;


/**
 * Patient medical history summary table model.
 *
 * @author Tim Anderson
 */
public class PatientHistoryTableModel extends AbstractPatientHistoryTableModel {

    /**
     * Constructs a {@link PatientHistoryTableModel}.
     *
     * @param context the layout context
     */
    public PatientHistoryTableModel(LayoutContext context) {
        this(context, DEFAULT_CACHE_SIZE);
    }

    /**
     * Constructs a {@link PatientHistoryTableModel}.
     *
     * @param context   the layout context
     * @param cacheSize the render cache size
     */
    public PatientHistoryTableModel(LayoutContext context, int cacheSize) {
        super(PatientArchetypes.CLINICAL_EVENT, context, cacheSize);
    }

    /**
     * Returns a component for the act type.
     * <p/>
     * This indents the type depending on the acts depth in the act hierarchy.
     *
     * @param bean the act
     * @return a component representing the act type
     */
    @Override
    protected Component getType(IMObjectBean bean) {
        Component result;
        if (bean.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
            result = getHyperlinkedType(bean);
        } else {
            result = super.getType(bean);
        }
        return result;
    }

    /**
     * Returns the name of an act to display in the Type column.
     *
     * @param bean the act
     * @return the name
     */
    @Override
    protected String getTypeName(IMObjectBean bean) {
        String result = super.getTypeName(bean);
        if (bean.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
            if (isOngoingProblem(bean)) {
                result = Messages.format("patient.record.summary.ongoingProblem", result);
            }
        }
        return result;
    }

    /**
     * Returns a component for a parent act.
     *
     * @param bean the parent act
     * @return a component representing the act
     * @throws OpenVPMSException for any error
     */
    @Override
    protected Component formatParent(IMObjectBean bean) {
        Component result;
        String date = formatDateRange(bean);
        String text = formatEventText(bean);
        Label summary = LabelFactory.create(null, Styles.BOLD);
        summary.setText(Messages.format("patient.record.summary.datedTitle", date, text));

        String detail = getText(bean.getObject(Act.class), false);
        if (detail != null) {
            Label detailLabel = getTextDetail(detail);

            // create some padding, to indent the detail to the same level as the type.
            Row padding = RowFactory.create(Styles.INSET, new Label(""));
            Component dateSpacer = getDateSpacer();

            result = new Column();
            result.add(summary);
            result.add(RowFactory.create(Styles.CELL_SPACING, padding, dateSpacer, detailLabel));
        } else {
            result = summary;
        }
        return result;
    }

    /**
     * Formats an act item.
     *
     * @param bean the item bean
     * @return a component representing the item
     */
    @Override
    protected Component formatItem(IMObjectBean bean) {
        Component detail;
        if (bean.isA(CustomerAccountArchetypes.INVOICE_ITEM)) {
            detail = getInvoiceItemDetail(bean);
        } else if (bean.isA(PatientArchetypes.CLINICAL_PROBLEM)) {
            detail = getProblemDetail(bean);
        } else {
            detail = super.formatItem(bean);
        }
        return detail;
    }

    /**
     * Returns a component for the detail of an <em>act.patientMedication</em>.
     *
     * @param bean        the act bean
     * @param showBatches if (@code true}, include any batch number
     */
    @Override
    protected Component getMedicationDetail(IMObjectBean bean, boolean showBatches) {
        String text = getText(bean.getObject(Act.class), true);
        List<Reference> refs = bean.getSourceRefs("invoiceItem");
        if (!refs.isEmpty()) {
            ArchetypeQuery query = new ArchetypeQuery(refs.get(0));
            query.getArchetypeConstraint().setAlias("act");
            query.add(new NodeSelectConstraint("total"));
            query.setMaxResults(1);
            ObjectSetQueryIterator iter = new ObjectSetQueryIterator(query);
            if (iter.hasNext()) {
                ObjectSet set = iter.next();
                text = Messages.format("patient.record.summary.medication", text,
                                       NumberFormatter.formatCurrency(set.getBigDecimal("act.total")));
            }
        }
        Component component = getTextDetail(text);
        if (showBatches) {
            component = addBatch(bean, component);
        }
        return component;
    }

    /**
     * Returns a component for the detail of an act.patientMedication.
     * <p/>
     * This includes the invoice item amount, if one is available.
     *
     * @param bean the act
     * @return a new component
     */
    private Component getInvoiceItemDetail(IMObjectBean bean) {
        String name = getName(bean.getTargetRef("product"));
        FinancialAct act = bean.getObject(FinancialAct.class);
        String text = Messages.format("patient.record.summary.invoiceitem", name, act.getQuantity(),
                                      NumberFormatter.formatCurrency(act.getTotal()));
        return getTextDetail(text);
    }

    /**
     * Returns a component for the detail of an act.patientClinicalProblem.
     *
     * @param bean the problem
     * @return a new component
     */
    private Component getProblemDetail(IMObjectBean bean) {
        Act act = bean.getObject(Act.class);
        Component result = getTextDetail(act);
        String presentingComplaint = LookupNameHelper.getName(act, "presentingComplaint");
        if (presentingComplaint != null) {
            Label label = LabelFactory.create();
            label.setText(Messages.format("patient.record.summary.presentingComplaint", presentingComplaint));
            result = ColumnFactory.create(Styles.CELL_SPACING, result, label);
        }
        return result;
    }

    /**
     * Determines if a problem is an ongoing one.
     *
     * @param bean the problem
     * @return {@code true} if the problem is ongoing
     */
    private boolean isOngoingProblem(IMObjectBean bean) {
        List<Reference> eventRefs = bean.getSourceRefs("events");
        if (eventRefs.size() > 1) {
            Act event = getParent(bean.getObject(Act.class));
            if (event != null) {
                eventRefs.remove(event.getObjectReference());
                return !isEarliestEvent(event, eventRefs);
            }
        }
        return false;
    }

    /*
     * Determines if an event is the earliest in a set of events.
     *
     * @param event the event
     * @param references the event references, excluding the event
     * @return {@code true} if the event has the earliest start time
     */
    private boolean isEarliestEvent(Act event, List<Reference> references) {
        boolean result = false;
        List<Long> dates = new ArrayList<>();
        // try and find all of the events in the list of objects first.
        for (Act object : getObjects()) {
            if (object.isA(PatientArchetypes.CLINICAL_EVENT)
                && references.remove(object.getObjectReference())) {
                dates.add(object.getActivityStartTime().getTime());
            }
        }
        if (references.isEmpty()) {
            // found all of the events
            Collections.sort(dates);
            result = event.getActivityStartTime().getTime() < dates.get(0);
        } else {
            // need to hit the database to find the earliest event
            Object[] ids = new Object[references.size()];
            for (int j = 0; j < references.size(); ++j) {
                ids[j] = references.get(j).getId();
            }
            ArchetypeQuery query = new ArchetypeQuery(shortName("event", PatientArchetypes.CLINICAL_EVENT));
            query.add(Constraints.sort("startTime"));
            query.add(new NodeSelectConstraint("event.startTime"));
            query.add(Constraints.in("id", ids));
            query.setMaxResults(1);
            ObjectSetQueryIterator iterator = new ObjectSetQueryIterator(query);
            if (iterator.hasNext()) {
                ObjectSet set = iterator.next();
                if (DateRules.compareTo(event.getActivityStartTime(), set.getDate("event.startTime")) < 0) {
                    result = true;
                }
            }
        }
        return result;
    }

}
