/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */
package org.openvpms.web.workspace.patient.charge;

import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.product.Product;
import org.openvpms.web.system.ServiceHelper;

import java.util.List;


/**
 * Tracks charge items added via the expansion of a <em>product.template</em>.
 *
 * @author Tim Anderson
 */
public class TemplateChargeItems {

    /**
     * The product.
     */
    private final Product product;

    /**
     * The charge items.
     */
    private final List<Act> chargeItems;

    /**
     * Constructs a {@code TemplateChargeItems}.
     *
     * @param product the template product
     * @param items   the charge items generated by the expansion of the template
     */
    public TemplateChargeItems(Product product, List<Act> items) {
        this.product = product;
        this.chargeItems = items;
    }

    /**
     * Returns the template product.
     *
     * @return the template product
     */
    public Product getProduct() {
        return product;
    }

    /**
     * Returns the visit note associated with the template product.
     *
     * @return the visit note. May be {@code null}
     */
    public String getVisitNote() {
        IMObjectBean bean = ServiceHelper.getArchetypeService().getBean(product);
        return bean.getString("visitNote");
    }

    /**
     * Finds the first template charge item present in the supplied items.
     *
     * @param items the items to search
     * @return the first template charge item, or {@code null} if none exists
     */
    public Act findFirst(List<Act> items) {
        for (Act item : items) {
            if (chargeItems.contains(item)) {
                return item;
            }
        }
        return null;
    }

}
