/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.customer.payment;


import org.openvpms.archetype.rules.customer.CustomerArchetypes;
import org.openvpms.archetype.rules.finance.account.CustomerAccountRules;
import org.openvpms.archetype.rules.finance.reminder.AccountReminderRules;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.domain.service.object.DomainObjectService;
import org.openvpms.paymentprocessor.transaction.Transaction;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.im.archetype.Archetypes;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.factory.ButtonFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.system.ServiceHelper;
import org.openvpms.web.workspace.customer.CustomerActCRUDWindow;
import org.openvpms.web.workspace.customer.account.AccountActActions;
import org.openvpms.web.workspace.workflow.payment.OpenDrawerTask;

import static org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus.SUBMITTED;


/**
 * CRUD window for payments.
 *
 * @author Tim Anderson
 */
public class PaymentCRUDWindow extends CustomerActCRUDWindow<FinancialAct> {

    /**
     * Payment processor send link button identifier.
     */
    private static final String SEND_LINK = "button.sendLink";

    /**
     * Constructs a {@code PaymentCRUDWindow}.
     *
     * @param archetypes the archetypes that this may create
     * @param context    the context
     * @param help       the help context
     */
    public PaymentCRUDWindow(Archetypes<FinancialAct> archetypes, Context context, HelpContext help) {
        super(archetypes, new PaymentActions(context), context, help);
    }

    /**
     * Invoked when posting of an act is complete. This pops up a dialog to print the act, and opens the cash drawer
     * if required.
     *
     * @param act the act
     */
    @Override
    protected void onPosted(FinancialAct act) {
        super.onPosted(act);
        OpenDrawerTask task = new OpenDrawerTask();
        task.open(act);
    }

    /**
     * Lays out the buttons.
     *
     * @param buttons the button row
     */
    @Override
    protected void layoutButtons(ButtonSet buttons) {
        super.layoutButtons(buttons);
        buttons.add(createPostButton());
        buttons.add(createPrintButton());
        buttons.add(createMailButton());
        buttons.add(ButtonFactory.create(SEND_LINK, action(this::sendLink)));
    }

    /**
     * Enables/disables the buttons that require an object to be selected.
     *
     * @param buttons the button set
     * @param enable  determines if buttons should be enabled
     */
    @Override
    protected void enableButtons(ButtonSet buttons, boolean enable) {
        super.enableButtons(buttons, enable);
        buttons.setEnabled(POST_ID, enable);
        enablePrintPreview(buttons, enable);
        buttons.setEnabled(SEND_LINK, enable && getActions().canSendLink(getObject()));
    }

    /**
     * Returns the actions that may be performed on the selected object.
     *
     * @return the actions
     */
    @Override
    protected PaymentActions getActions() {
        return (PaymentActions) super.getActions();
    }

    /**
     * Sends a payment processor link.
     *
     * @param act the payment/refund
     */
    private void sendLink(FinancialAct act) {
        PaymentActions actions = getActions();
        FinancialAct transactionAct = actions.getPaymentProcessorTransaction(act);
        if (transactionAct != null && actions.canSendLinkForTransaction(transactionAct)) {
            DomainObjectService service = ServiceHelper.getBean(DomainObjectService.class);
            Transaction transaction = service.create(transactionAct, Transaction.class);
            PaymentProcessorStatusDialog dialog = PaymentProcessorStatusDialog.create(
                    transaction.getPaymentProcessor(), transaction, transactionAct,
                    createLayoutContext(getHelpContext()), true);
            dialog.show();
        } else {
            onRefresh(act);
        }
    }

    protected static class PaymentActions extends AccountActActions {

        /**
         * The context.
         */
        private final Context context;

        /**
         * Constructs a {@link PaymentActions}.
         *
         * @param context the context
         */
        public PaymentActions(Context context) {
            super(ServiceHelper.getBean(CustomerAccountRules.class), ServiceHelper.getBean(AccountReminderRules.class),
                  ServiceHelper.getBean(PracticeService.class), ServiceHelper.getArchetypeService());
            this.context = context;
        }

        /**
         * Determines if objects can be created.
         *
         * @return {@code true} if there is a customer of type <em>party.customerperson</em> otherwise {@code false}.
         * This prevents payments being created for OTC customers.
         */
        @Override
        public boolean canCreate() {
            Party customer = context.getCustomer();
            return customer != null && customer.isA(CustomerArchetypes.PERSON);
        }

        /**
         * Determines if a payment processor link can be sent.
         *
         * @param act the payment/refund
         * @return {@code true} if a payment processor link can be sent, otherwise {@code false}
         */
        public boolean canSendLink(FinancialAct act) {
            FinancialAct transaction = getPaymentProcessorTransaction(act);
            return canSendLinkForTransaction(transaction);
        }

        /**
         * Determines if a payment processor link can be sent.
         *
         * @param transaction the payment processor transaction
         * @return {@code true} if a payment processor link can be sent, otherwise {@code false}
         */
        public boolean canSendLinkForTransaction(FinancialAct transaction) {
            boolean result = false;
            if (isSubmitted(transaction)) {
                String url = getBean(transaction).getString("url");
                result = url != null;
            }
            return result;
        }

        /**
         * Determines if a payment processor transaction is submitted.
         *
         * @param transaction the transaction
         * @return {@code true} if the transaction is submitted, otherwise {@code false}
         */
        public boolean isSubmitted(FinancialAct transaction) {
            return (transaction != null && SUBMITTED.equals(transaction.getStatus()));
        }

        /**
         * Returns the payment processor transaction in a payment or refund.
         *
         * @param act the payment/refund
         * @return the most recent payment processor transaction, or {@code null} if there is none
         */
        public FinancialAct getPaymentProcessorTransaction(FinancialAct act) {
            return getCustomerAccountRules().getPaymentProcessorTransaction(act);
        }
    }
}
