/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.workspace.admin.template.paymentprocessor;

import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.web.component.app.Context;
import org.openvpms.web.component.app.LocalContext;
import org.openvpms.web.component.im.sms.SMSTemplateEvaluator;
import org.openvpms.web.component.macro.MacroVariables;

/**
 * Expression evaluator for payment processor SMS.
 *
 * @author Tim Anderson
 */
public class PaymentProcessorSMSEvaluator {

    /**
     * The practice.
     */
    private final Party practice;

    /**
     * The template evaluator.
     */
    private final SMSTemplateEvaluator evaluator;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The lookup service.
     */
    private final LookupService lookups;

    /**
     * Constructs an {@link PaymentProcessorSMSEvaluator}.
     *
     * @param practice  the practice
     * @param evaluator the evaluator
     * @param service   the service
     * @param lookups   the lookup service
     */
    public PaymentProcessorSMSEvaluator(Party practice, SMSTemplateEvaluator evaluator, ArchetypeService service,
                                        LookupService lookups) {
        this.practice = practice;
        this.evaluator = evaluator;
        this.service = service;
        this.lookups = lookups;
    }

    /**
     * Evaluates a payment processor SMS template against a transaction.
     * <p/>
     * The following variables are defined:
     * <ul>
     *     <li>$customer - the customer the transaction is for. A <em>party.customerperson</em></li>
     *     <li>$location - the practice location where the transaction was performed. A
     *     <em>party.organisationPractice</em></li>
     *     <li>$practice - a <em>party.organisationPractice</em></li>
     *     <li>$url - the customer transaction url, specified by the payment processor</li>
     *     <li>$description - the transaction description</li>
     *     <li>$amount - the transaction amount</li>
     *     <li>$processor - the name of the payment processor</li>
     * </ul>
     *
     * @param template    the template
     * @param transaction the payment processor transaction
     * @return the result of the expression
     */
    public String evaluate(Entity template, FinancialAct transaction) {
        String result;
        IMObjectBean bean = service.getBean(transaction);
        Party customer = bean.getTarget("customer", Party.class);
        Party location = bean.getTarget("location", Party.class);
        Entity processor = bean.getTarget("paymentProcessor", Entity.class);

        Context context = new LocalContext();
        context.setCustomer(customer);
        context.setLocation(location);
        context.setPractice(practice);
        MacroVariables variables = new MacroVariables(context, service, lookups);

        variables.add("txn", transaction);
        variables.add("url", bean.getString("url"));
        variables.add("processor", processor != null ? processor.getName() : null);
        variables.add("amount", transaction.getTotal());
        variables.add("description", transaction.getDescription());
        result = evaluator.evaluate(template, transaction, variables);
        return result;
    }
}