/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.jobs.appointment;

import org.apache.commons.collections4.ComparatorUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Period;
import org.openvpms.archetype.rules.party.CustomerRules;
import org.openvpms.archetype.rules.patient.PatientRules;
import org.openvpms.archetype.rules.practice.LocationRules;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.rules.util.PeriodHelper;
import org.openvpms.archetype.rules.workflow.AppointmentRules;
import org.openvpms.archetype.rules.workflow.ScheduleArchetypes;
import org.openvpms.archetype.rules.workflow.SystemMessageReason;
import org.openvpms.component.business.service.archetype.rule.IArchetypeRuleService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.user.User;
import org.openvpms.component.system.common.query.IPage;
import org.openvpms.component.system.common.query.NamedQuery;
import org.openvpms.component.system.common.query.ObjectSet;
import org.openvpms.sms.exception.SMSException;
import org.openvpms.web.component.service.SimpleSMSService;
import org.openvpms.web.jobs.JobCompletionNotifier;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.resource.i18n.format.DateFormatter;
import org.openvpms.web.workspace.workflow.appointment.reminder.AppointmentReminderEvaluator;
import org.openvpms.web.workspace.workflow.appointment.reminder.AppointmentReminderException;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.InterruptableJob;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.Trigger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * A job that sends SMS appointment reminders.
 * <p/>
 * It is configured by an <em>entity.jobAppointmentReminder</em>.
 *
 * @author Tim Anderson
 */
@DisallowConcurrentExecution
public class AppointmentReminderJob implements InterruptableJob {

    /**
     * The job configuration.
     */
    private final Entity configuration;

    /**
     * The SMS service.
     */
    private final SimpleSMSService service;

    /**
     * The archetype service.
     */
    private final IArchetypeRuleService archetypeService;

    /**
     * The appointment rules.
     */
    private final AppointmentRules appointmentRules;

    /**
     * The customer rules.
     */
    private final CustomerRules customerRules;

    /**
     * The patient rules.
     */
    private final PatientRules patientRules;

    /**
     * The practice service.
     */
    private final PracticeService practiceService;

    /**
     * The location rules.
     */
    private final LocationRules locationRules;

    /**
     * The appointment reminder evaluator.
     */
    private final AppointmentReminderEvaluator evaluator;

    /**
     * The transaction manager.
     */
    private final PlatformTransactionManager transactionManager;

    /**
     * The interval prior to an appointment when reminders can be sent.
     */
    private final Period fromPeriod;

    /**
     * The interval prior to an appointment when reminders should no longer be sent.
     */
    private final Period toPeriod;

    /**
     * Communications logging subject.
     */
    private final String subject;

    /**
     * Used to send messages to users on completion or failure.
     */
    private final JobCompletionNotifier notifier;

    /**
     * The maximum no. of message parts supported by the provider.
     */
    private final int maxParts;

    /**
     * Schedules that failed to have reminders sent, and the corresponding dates, ordered on schedule name.
     */
    private final Map<Entity, Set<Date>> errors = new TreeMap<>(
            (o1, o2) -> ComparatorUtils.nullLowComparator(null).compare(o1.getName(), o2.getName()));

    /**
     * Determines if reminding should stop.
     */
    private volatile boolean stop;

    /**
     * Determines the minimum appointment start time.
     */
    private Date minStartTime;

    /**
     * Determines the maximum appointment start time.
     */
    private Date maxStartTime;

    /**
     * The total no. of reminders processed.
     */
    private int total;

    /**
     * The no. of reminders sent.
     */
    private int sent;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(AppointmentReminderJob.class);

    /**
     * Communication logging reason code.
     */
    private static final String REASON = "APPOINTMENT_REMINDER";

    /**
     * Constructs an {@link AppointmentReminderJob}.
     *
     * @param configuration      the job configuration
     * @param service            the SMS service
     * @param archetypeService   the archetype service
     * @param appointmentRules   the appointment rules
     * @param customerRules      the customer rules
     * @param patientRules       the patient rules
     * @param practiceService    the practice service
     * @param locationRules      the location rules
     * @param evaluator          the appointment reminder evaluator
     * @param transactionManager the transaction manager
     */
    public AppointmentReminderJob(Entity configuration, SimpleSMSService service,
                                  IArchetypeRuleService archetypeService, AppointmentRules appointmentRules,
                                  CustomerRules customerRules, PatientRules patientRules,
                                  PracticeService practiceService, LocationRules locationRules,
                                  AppointmentReminderEvaluator evaluator,
                                  PlatformTransactionManager transactionManager) {
        this.configuration = configuration;
        this.service = service;
        this.archetypeService = archetypeService;
        this.appointmentRules = appointmentRules;
        this.customerRules = customerRules;
        this.patientRules = patientRules;
        this.practiceService = practiceService;
        this.locationRules = locationRules;
        this.evaluator = evaluator;
        this.transactionManager = transactionManager;
        maxParts = service.getMaxParts();
        IMObjectBean bean = archetypeService.getBean(configuration);
        fromPeriod = PeriodHelper.getPeriod(bean, "smsFrom", "smsFromUnits", DateUnits.WEEKS);
        toPeriod = PeriodHelper.getPeriod(bean, "smsTo", "smsToUnits", DateUnits.DAYS);
        notifier = new JobCompletionNotifier(archetypeService);
        subject = Messages.get("sms.log.appointment.subject");
    }

    /**
     * Called by the {@link Scheduler} when a user interrupts the {@code Job}.
     */
    @Override
    public void interrupt() {
        stop = true;
    }

    /**
     * Called by the {@link Scheduler} when a {@link Trigger} fires that is associated with the {@code Job}.
     *
     * @throws JobExecutionException if there is an exception while executing the job.
     */
    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        try {
            execute();
            complete(null);
        } catch (Throwable exception) {
            log.error(exception.getMessage(), exception);
            complete(exception);
        }
    }

    /**
     * Sends appointment reminders.
     */
    protected void execute() {
        total = 0;
        sent = 0;
        Party practice = practiceService.getPractice();
        if (practice == null) {
            throw new IllegalStateException("No current practice");
        }
        List<Party> locations = practiceService.getLocations();
        Entity defaultTemplate = practiceService.getAppointmentSMSTemplate();
        Map<Party, Entity> templates = new HashMap<>();
        for (Party location : locations) {
            addTemplate(location, templates);
        }
        if (defaultTemplate == null && templates.isEmpty()) {
            throw new IllegalStateException("No Appointment Reminder SMS Templates have been configured");
        }
        NamedQuery query = new NamedQuery("AppointmentReminderJob.getReminders", "id");
        Date date = getStartDate();
        minStartTime = DateRules.plus(date, toPeriod);
        maxStartTime = DateRules.plus(date, fromPeriod);

        if (log.isInfoEnabled()) {
            log.info("Sending reminders for appointments between " + DateFormatter.formatDateTime(minStartTime)
                     + " and " + DateFormatter.formatDateTime(maxStartTime));
        }

        query.setParameter("from", minStartTime);
        query.setParameter("to", maxStartTime);
        int pageSize = getPageSize();
        query.setMaxResults(pageSize);
        // pull in count results at a time. Note that sending updates the appointment, which affects paging, so the
        // query needs to be re-issued from the start if any have updated
        boolean done = false;
        Set<Long> exclude = new HashSet<>();
        while (!stop && !done) {
            IPage<ObjectSet> page = archetypeService.getObjects(query);
            boolean updated = false;  // flag to indicate if any reminders were updated
            for (ObjectSet set : page.getResults()) {
                long id = set.getLong("id");
                IMObjectBean bean = getAppointment(id);
                if (bean != null && !exclude.contains(id) && canSend(bean)) {
                    ++total;
                    if (send(bean, practice, defaultTemplate, templates)) {
                        ++sent;
                        updated = true;
                    } else {
                        // failed to send the reminder, so flag the act for exclusion if a query retrieves it again
                        exclude.add(id);
                    }
                }
            }
            if (page.getResults().size() < pageSize) {
                done = true;
            } else if (!updated) {
                // nothing updated, so pull in the next page
                query.setFirstResult(query.getFirstResult() + page.getResults().size());
            }
        }
        if (log.isInfoEnabled()) {
            log.info("Sent " + sent + " of " + total + " appointment reminders");
        }
    }

    /**
     * Returns the date/time to base date calculations on.
     *
     * @return the current date/time
     */
    protected Date getStartDate() {
        return new Date();
    }

    /**
     * Determines the no. of appointments to process at once.
     *
     * @return the page size
     */
    protected int getPageSize() {
        return 1000;
    }

    /**
     * Sets an appointment reminder SMS.
     *
     * @param bean            the appointment
     * @param practice        the practice
     * @param defaultTemplate the default reminder template
     * @param templates       the location specific reminder templates
     * @return {@code true} if the reminder was sent
     * @throws SMSException if the send fails
     */
    protected boolean send(IMObjectBean bean, Party practice, Entity defaultTemplate, Map<Party, Entity> templates) {
        boolean sent = false;
        Party customer = bean.getTarget("customer", Party.class);
        if (customer != null && isCustomerValid(customer, bean) && isPatientValid(bean)) {
            Contact contact = customerRules.getSMSContact(customer);
            if (contact == null) {
                addError(bean, Messages.get("sms.appointment.nocontact"));
            } else {
                Party location = getLocation(bean);
                if (location == null) {
                    addError(bean, Messages.format("reporting.reminder.nolocation", bean.getDisplayName()));
                } else {
                    Entity template = templates.get(location);
                    if (template == null) {
                        template = defaultTemplate;
                    }
                    if (template == null) {
                        addError(bean, Messages.format("sms.appointment.notemplate", location.getName()));
                    } else {
                        sent = send(bean, practice, customer, contact, location, template);
                    }
                }
            }
        }
        return sent;
    }

    /**
     * Determines if a reminder can be sent for an appointment.
     *
     * @param bean the appointment bean
     * @return {@code true} if the startTime isn't in the past, a reminder is flagged to be sent, and no reminder has
     * already been sent
     */
    protected boolean canSend(IMObjectBean bean) {
        return !isPast(bean) && bean.getBoolean("sendReminder") && bean.getDate("reminderSent") == null;
    }

    /**
     * Determines if an appointment is past.
     *
     * @param bean the appointment bean
     * @return {@code true} if the appointment is past
     */
    protected boolean isPast(IMObjectBean bean) {
        return DateRules.compareTo(bean.getDate("startTime"), new Date()) <= 0;
    }

    /**
     * Returns the location associated with an appointment.
     *
     * @param bean the appointment bean
     * @return the location, or {@code null} if none can be determined
     */
    protected Party getLocation(IMObjectBean bean) {
        Party location = null;
        Entity schedule = bean.getTarget("schedule", Entity.class);
        if (schedule != null) {
            IMObjectBean scheduleBean = archetypeService.getBean(schedule);
            location = (Party) scheduleBean.getTarget("location", Policies.active());
            if (location == null) {
                log.warn("Cannot determine the practice location for: " + schedule.getName());
            }
        }
        return location;
    }

    /**
     * Sets an appointment reminder SMS.
     *
     * @param bean     the appointment
     * @param practice the practice
     * @param customer the customer
     * @param contact  the contact
     * @param location the location
     * @param template the template
     * @return {@code true} if the reminder was sent
     * @throws SMSException if the send fails
     */
    private boolean send(IMObjectBean bean, Party practice, Party customer, Contact contact, Party location,
                         Entity template) {
        boolean sent = false;
        Act appointment = bean.getObject(Act.class);
        try {
            String message = evaluator.evaluate(template, appointment, location, practice);
            if (StringUtils.isEmpty(message)) {
                addError(bean, Messages.get("reporting.reminder.emptysms"));
            } else if (service.getParts(message) > maxParts) {
                addError(bean, Messages.format("reporting.reminder.smstoolong", message));
            } else {
                TransactionTemplate txnTemplate = new TransactionTemplate(transactionManager);
                txnTemplate.executeWithoutResult(transactionStatus -> {
                    // execute in a transaction so the SMS rolls back if the reminder cannot be updated
                    Party patient = bean.getTarget("patient", Party.class);
                    service.send(message, contact, customer, patient, subject, REASON, location, appointment);
                    appointmentRules.setSMSReminderSent(appointment, new Date());
                });
                sent = true;
            }
        } catch (AppointmentReminderException exception) {
            log.error(exception.getMessage(), exception);
            IMObjectBean reloaded = getAppointment(appointment.getId());
            // reload the appointment as any relationship to an SMS will be invalid and prevent subsequent save
            if (reloaded != null) {
                addError(reloaded, exception.getMessage());
            }
        }
        return sent;
    }

    /**
     * Determines if the appointment customer is valid.
     *
     * @param customer the customer
     * @param bean     the appointment bean
     * @return {@code true} if the customer is valid
     */
    private boolean isCustomerValid(Party customer, IMObjectBean bean) {
        boolean valid = customer.isActive();
        if (!valid) {
            addError(bean, Messages.get("sms.appointment.customerinactive"));
        }
        return valid;
    }

    /**
     * Determines if the appointment patient is valid. It is valid if no patient is present, or it is active and
     * not deceased.
     *
     * @param bean the appointment bean
     * @return {@code true} if the patient is valid
     */
    private boolean isPatientValid(IMObjectBean bean) {
        boolean valid;
        Party patient = bean.getTarget("patient", Party.class);
        if (patient == null) {
            valid = true;
        } else if (patientRules.isDeceased(patient)) {
            addError(bean, Messages.get("sms.appointment.patientdeceased"));
            valid = false;
        } else if (!patient.isActive()) {
            addError(bean, Messages.get("sms.appointment.patientinactive"));
            valid = false;
        } else {
            valid = true;
        }
        return valid;
    }

    /**
     * Returns an appointment given its identifier.
     *
     * @param id the appointment identifier
     * @return the appointment, or {@code null} if it doesn't exist
     */
    private IMObjectBean getAppointment(long id) {
        Act act = archetypeService.get(ScheduleArchetypes.APPOINTMENT, id, Act.class);
        return act != null ? archetypeService.getBean(act) : null;
    }

    /**
     * Adds an error.
     * <p/>
     * This populates the reminderError node, ensuring the contents don't exceed the maximum length, and
     * logs the schedule and date of the reminder for reporting by {@link #notifyUsers}.
     *
     * @param bean    the reminder bean
     * @param message the error message
     */
    private void addError(IMObjectBean bean, String message) {
        log.error("Failed to send reminder, id=" + bean.getObject().getId() + ", message=" + message);
        int maxLength = bean.getMaxLength("reminderError");
        bean.setValue("reminderError", StringUtils.abbreviate(message, maxLength));
        bean.save();
        Entity schedule = bean.getTarget("schedule", Entity.class);
        if (schedule != null) {
            Set<Date> dates = errors.computeIfAbsent(schedule, k -> new TreeSet<>());
            dates.add(DateRules.getDate(bean.getDate("startTime")));
        }
    }

    /**
     * Adds a template for a location to the supplied cache, if one exists.
     *
     * @param location  the practice location
     * @param templates the template cache
     */
    private void addTemplate(Party location, Map<Party, Entity> templates) {
        Entity template = locationRules.getAppointmentSMSTemplate(location);
        if (template != null) {
            templates.put(location, template);
        }
    }

    /**
     * Invoked on completion of a job. Sends a message notifying the registered users of completion or failure of the
     * job if required.
     *
     * @param exception the exception, if the job failed, otherwise {@code null}
     */
    private void complete(Throwable exception) {
        if (exception != null || sent != 0 || total != 0) {
            Set<User> users = notifier.getUsers(configuration);
            if (!users.isEmpty()) {
                notifyUsers(users, exception);
            }
        }
    }

    /**
     * Notifies users of completion or failure of the job.
     *
     * @param users     the users to notify
     * @param exception the exception, if the job failed, otherwise {@code null}
     */
    private void notifyUsers(Set<User> users, Throwable exception) {
        String subject;
        String reason;
        StringBuilder text = new StringBuilder();
        if (exception != null) {
            reason = SystemMessageReason.ERROR;
            subject = Messages.format("appointmentreminder.subject.exception", configuration.getName());
            text.append(Messages.format("appointmentreminder.exception", exception.getMessage()));
        } else {
            if (sent != total || !errors.isEmpty()) {
                reason = SystemMessageReason.ERROR;
                subject = Messages.format("appointmentreminder.subject.errors", configuration.getName(), total - sent);
            } else {
                reason = SystemMessageReason.COMPLETED;
                subject = Messages.format("appointmentreminder.subject.success", configuration.getName(), sent);
            }
        }
        if (minStartTime != null && maxStartTime != null) {
            text.append(Messages.format("appointmentreminder.period", DateFormatter.formatDateTime(minStartTime),
                                        DateFormatter.formatDateTime(maxStartTime)));
            text.append("\n");
        }
        text.append(Messages.format("appointmentreminder.sent", sent, total));

        if (!errors.isEmpty()) {
            text.append("\n\n");
            text.append(Messages.get("appointmentreminder.error"));
            text.append("\n");
            for (Map.Entry<Entity, Set<Date>> entry : errors.entrySet()) {
                for (Date date : entry.getValue()) {
                    text.append(Messages.format("appointmentreminder.error.item", entry.getKey().getName(),
                                                DateFormatter.formatDate(date, false)));
                    text.append("\n");
                }
            }
        }
        notifier.send(users, subject, reason, text.toString());
    }
}
