/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.echo.dialog;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Builder for {@link OptionDialog}s.
 *
 * @author Tim Anderson
 */
public class OptionDialogBuilder extends MessageDialogBuilder<OptionDialog, OptionDialogBuilder> {

    /**
     * The options.
     */
    private final Map<String, Runnable> options = new LinkedHashMap<>();

    /**
     * The selected index.
     */
    private int selected = -1;

    /**
     * Determines if the dialog closes on selection.
     */
    private boolean closeOnSelection;

    /**
     * Builds the dialog.
     *
     * @return the dialog
     */
    @Override
    public OptionDialog build() {
        return new OptionDialog(this);
    }

    /**
     * Adds an option.
     *
     * @param option the option text
     * @return this
     */
    public OptionDialogBuilder addOption(String option) {
        return addOption(option, null);
    }

    /**
     * Adds an option.
     *
     * @param option   the option text
     * @param listener the listener to invoke at dialog close, if the option is selected
     * @return this
     */
    public OptionDialogBuilder addOption(String option, Runnable listener) {
        this.options.put(option, listener);
        return this;
    }

    /**
     * Sets the selected option.
     *
     * @param index the option index
     * @return
     */
    public OptionDialogBuilder selected(int index) {
        this.selected = index;
        return this;
    }

    /**
     * Indicates to close the dialog when an option is selected.
     *
     * @return this
     */
    public OptionDialogBuilder closeOnSelection() {
        this.closeOnSelection = true;
        return this;
    }

    /**
     * Returns the options.
     *
     * @return the options, in the order they were added
     */
    public Map<String, Runnable> getOptions() {
        return options;
    }

    /**
     * Returns the selected option.
     *
     * @return the selected option, or {@code -1} if none is selected
     */
    public int getSelected() {
        return selected;
    }

    /**
     * Determines if the dialog closes on selection of an option.
     *
     * @return {@code true} if the dialog closes, {@code false} if it should remain open
     */
    public boolean isCloseOnSelection() {
        return closeOnSelection;
    }

    /**
     * Returns this.
     *
     * @return this
     */
    @Override
    protected OptionDialogBuilder getThis() {
        return this;
    }
}
