/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.echo.dialog;

import nextapp.echo2.app.Column;
import nextapp.echo2.app.Label;
import nextapp.echo2.app.RadioButton;
import nextapp.echo2.app.button.ButtonGroup;
import nextapp.echo2.app.event.ActionEvent;
import org.openvpms.web.echo.event.ActionListener;
import org.openvpms.web.echo.factory.ButtonFactory;
import org.openvpms.web.echo.factory.ColumnFactory;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.style.Styles;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * A dialog that prompts to select one of several options.
 *
 * @author Tim Anderson
 */
public class OptionDialog extends MessageDialog {

    /**
     * The options.
     */
    private final List<Option> options = new ArrayList<>();

    /**
     * Determines if the dialog should close on selection.
     */
    private boolean closeOnSelection = false;

    /**
     * Option listener.
     */
    private final ActionListener listener;

    /**
     * Construct an {@link OptionDialog}.
     *
     * @param title   the dialog title
     * @param message the message to display
     * @param options the options to display
     */
    public OptionDialog(String title, String message, String[] options) {
        this(title, message, options, (HelpContext) null);
    }

    /**
     * Construct an {@link OptionDialog}.
     *
     * @param title   the dialog title
     * @param message the message to display
     * @param options the options to display
     * @param buttons the buttons to display
     */
    public OptionDialog(String title, String message, String[] options, String[] buttons) {
        this(title, message, options, buttons, null);
    }

    /**
     * Construct an {@link OptionDialog}.
     *
     * @param title   the dialog title
     * @param message the message to display
     * @param options the options to display
     * @param help    the help context
     */
    public OptionDialog(String title, String message, String[] options, HelpContext help) {
        this(title, message, options, OK_CANCEL, help);
    }

    /**
     * Constructs an {@link OptionDialog}.
     *
     * @param title   the dialog title
     * @param message the message to display
     * @param options the options to display
     * @param buttons the dialog buttons to display
     * @param help    the help context
     */
    public OptionDialog(String title, String message, String[] options, String[] buttons, HelpContext help) {
        super(title, message, buttons, help);
        listener = new ActionListener() {
            @Override
            public void onAction(ActionEvent event) {
                onSelected();
            }
        };

        ButtonGroup group = new ButtonGroup();
        for (String option : options) {
            addOption(group, option, null);
        }
    }

    /**
     * Constructs an {@link OptionDialog}.
     *
     * @param builder the builder
     */
    public OptionDialog(OptionDialogBuilder builder) {
        super(builder);
        listener = new ActionListener() {
            @Override
            public void onAction(ActionEvent event) {
                onSelected();
            }
        };
        ButtonGroup group = new ButtonGroup();
        for (Map.Entry<String, Runnable> option : builder.getOptions().entrySet()) {
            addOption(group, option.getKey(), option.getValue());
        }
        if (builder.getSelected() != -1) {
            setSelected(builder.getSelected());
        }
        setCloseOnSelection(builder.isCloseOnSelection());
    }

    /**
     * Selects an option.
     *
     * @param selected the option offset, or -1 to deselect any existing option.
     */
    public void setSelected(int selected) {
        if (selected >= 0 && selected < options.size()) {
            options.get(selected).button.setSelected(true);
        } else {
            for (Option option : options) {
                option.button.setSelected(false);
            }
        }
    }

    /**
     * Returns the option button.
     *
     * @param index the option index
     * @return the option button, or {@code null} if the index is invalid
     */
    public RadioButton getOption(int index) {
        return (index >= 0 && index < options.size()) ? options.get(index).button : null;
    }

    /**
     * Returns the selected option.
     *
     * @return the selected option, or {@code -1} if no option is selected
     */
    public int getSelected() {
        int selected = -1;
        for (int i = 0; i < options.size(); ++i) {
            if (options.get(i).button.isSelected()) {
                selected = i;
                break;
            }
        }
        return selected;
    }

    /**
     * Determines if the dialog should close when an option is selected.
     *
     * @param close if {@code true}, close the dialog when an option is selected.
     */
    public void setCloseOnSelection(boolean close) {
        if (close != closeOnSelection) {
            closeOnSelection = close;
            for (Option option : options) {
                RadioButton button = option.button;
                if (close) {
                    button.addActionListener(listener);
                } else {
                    button.removeActionListener(listener);
                }
            }
        }
    }

    /**
     * Creates a builder for a new dialog.
     *
     * @return the builder
     */
    public static OptionDialogBuilder newDialog() {
        return new OptionDialogBuilder();
    }

    /**
     * Invoked when the 'OK' button is pressed. If an option is selected, this sets the action and closes the window.
     */
    @Override
    protected void onOK() {
        int selected = getSelected();
        if (selected != -1) {
            super.onOK(); // make sure the dialog closes first, as any listener may be asynchronous
            Option option = options.get(selected);
            if (option.listener != null) {
                runProtected(option.listener);
            }
        }
    }

    /**
     * Lays out the component prior to display.
     */
    @Override
    protected void doLayout() {
        Label message = LabelFactory.create(true, true);
        message.setText(getMessage());
        Column column = ColumnFactory.create(Styles.WIDE_CELL_SPACING, message);
        for (Option option : options) {
            column.add(option.button);
        }
        getLayout().add(ColumnFactory.create(Styles.LARGE_INSET, column));
    }

    /**
     * Adds an option.
     *
     * @param group    the button group
     * @param text     the option text
     * @param listener the listener to invoke at dialog close, if the option is selected. May be {@code null}
     */
    private void addOption(ButtonGroup group, String text, Runnable listener) {
        RadioButton button = ButtonFactory.create(null, group);
        button.setText(text);
        group.addButton(button);
        options.add(new Option(button, listener));
    }

    /**
     * Invoked when an option is selected.
     */
    private void onSelected() {
        if (closeOnSelection) {
            onOK();
        }
    }

    private static class Option {

        private final RadioButton button;

        private final Runnable listener;

        public Option(RadioButton button, Runnable listener) {
            this.button = button;
            this.listener = listener;
        }
    }
}
