/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.echo.dialog;

import nextapp.echo2.app.ApplicationInstance;
import nextapp.echo2.app.Button;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.SplitPane;
import nextapp.echo2.app.Window;
import nextapp.echo2.app.WindowPane;
import org.openvpms.web.echo.button.ButtonRow;
import org.openvpms.web.echo.button.ButtonSet;
import org.openvpms.web.echo.factory.SplitPaneFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.focus.FocusHelper;
import org.openvpms.web.echo.keyboard.KeyStrokeHelper;
import org.openvpms.web.echo.pane.ContentPane;
import org.openvpms.web.echo.util.StyleSheetHelper;


/**
 * Generic popup window.
 *
 * @author Tim Anderson
 */
public abstract class PopupWindow extends WindowPane {

    /**
     * The layout pane.
     */
    private final SplitPane layout;

    /**
     * The button row.
     */
    private final ButtonRow row;

    /**
     * The focus group.
     */
    private final FocusGroup focusGroup;

    /**
     * The root pane.
     */
    private ContentPane root;

    /**
     * The default button.
     */
    private String defaultButton;

    /**
     * Determines if the component has been laid out.
     */
    private boolean doneLayout = false;

    /**
     * Constructs a {@link PopupWindow}.
     *
     * @param title the window title
     */
    public PopupWindow(String title) {
        this(title, null, null);
    }

    /**
     * Constructs a {@link PopupWindow}.
     *
     * @param title the window title. May be {@code null}
     * @param style the window style. May be {@code null}
     * @param focus the focus group. May be {@code null}
     */
    public PopupWindow(String title, String style, FocusGroup focus) {
        super(title, null, null);
        if (style == null) {
            style = "PopupWindow";
        }
        setStyleName(style);
        focusGroup = new FocusGroup(getClass().getName());
        if (focus != null) {
            focusGroup.add(focus);
        }

        row = new ButtonRow(focusGroup, "DialogButtonRow", ButtonRow.BUTTON_STYLE);

        root = new ContentPane();
        layout = createSplitPane();
        layout.add(row);
        root.add(layout);
        add(root);
    }

    /**
     * Sets the content width.
     * <p/>
     * This is the area usable for user content.
     *
     * @param width the width, in pixels
     */
    public void setContentWidth(int width) {
        int adjust = StyleSheetHelper.getProperty("dialog.margins", 0);
        setWidth(new Extent(width + adjust));
    }

    /**
     * Sets the content height.
     * <p/>
     * This is the area usable for user content.
     *
     * @param height the height, in pixels
     */
    public void setContentHeight(int height) {
        int adjust = StyleSheetHelper.getProperty("dialog.title.height", 0)
                     + StyleSheetHelper.getProperty("splitpane-buttonrow.separatorPosition", 0)
                     + StyleSheetHelper.getProperty("dialog.margins", 0);
        setHeight(new Extent(height + adjust));
    }

    /**
     * Show the window.
     */
    public void show() {
        show(false);
    }

    /**
     * Show the window.
     *
     * @param top if {@code true}, the dialog will display above other dialogs
     */
    public void show(boolean top) {
        if (!doneLayout) {
            doLayout();
            doneLayout = true;
        }
        if (getParent() == null) {
            DialogManager.show(this, top);
        }
        if (defaultButton != null) {
            getButtons().setFocus(defaultButton);
        }
    }

    /**
     * Sets the default button identifier.
     *
     * @param id the button identifier
     */
    public void setDefaultButton(String id) {
        defaultButton = id;
    }

    /**
     * Returns the default button identifier.
     *
     * @return the default button, or {@code null} if none has been set
     */
    public String getDefaultButton() {
        return defaultButton;
    }

    /**
     * Close the window.
     */
    public void close() {
        if (getParent() != null) {
            userClose();
        }
    }

    /**
     * Returns the buttons.
     *
     * @return the buttons
     */
    public ButtonSet getButtons() {
        return row.getButtons();
    }

    /**
     * Processes a user request to close the window (via the close button).
     * NOTE: This re-implements the super.userClose() method to avoid invoking getParent().remove(this)
     * if there is no parent.
     */
    @Override
    public void userClose() {
        fireWindowClosing();
        Integer defaultCloseOperationValue = (Integer) getRenderProperty(PROPERTY_DEFAULT_CLOSE_OPERATION);
        int defaultCloseOperation = defaultCloseOperationValue == null
                                    ? DISPOSE_ON_CLOSE : defaultCloseOperationValue;
        switch (defaultCloseOperation) {
            case DISPOSE_ON_CLOSE:
                if (getParent() != null) {
                    getParent().remove(this);
                }
                break;
            case HIDE_ON_CLOSE:
                setVisible(false);
                break;
        }
    }

    /**
     * Returns the root content pane.
     *
     * @return the content pane
     */
    protected ContentPane getContentPane() {
        return root;
    }

    /**
     * Creates the layout split pane.
     *
     * @return a new split pane
     */
    protected SplitPane createSplitPane() {
        return SplitPaneFactory.create(SplitPane.ORIENTATION_VERTICAL_BOTTOM_TOP, "PopupWindow.Layout");
    }

    /**
     * Lays out the component prior to display.
     * This implementation is a no-op.
     */
    protected void doLayout() {
    }

    /**
     * Returns the layout pane.
     *
     * @return the layout pane
     */
    protected SplitPane getLayout() {
        return layout;
    }

    /**
     * Adds a button.
     *
     * @param id       the button identifier
     * @param listener the listener
     * @return a new button
     */
    protected Button addButton(String id, Runnable listener) {
        return addButton(id, false, listener);
    }

    /**
     * Adds a button.
     *
     * @param id              the button identifier
     * @param disableShortcut if {@code true} disable any keyboard shortcut
     * @param listener        the listener
     * @return a new button
     */
    protected Button addButton(String id, boolean disableShortcut, Runnable listener) {
        return row.addButton(id, disableShortcut, listener);
    }

    /**
     * Returns the focus group.
     *
     * @return the focus group
     */
    protected FocusGroup getFocusGroup() {
        return focusGroup;
    }

    /**
     * Notifies {@code WindowPaneListener}s that the user has requested to close this {@code WindowPane}.
     * <p>
     * This implementation re-registers keystroke listeners as a workaround to bugs in Firefox.
     *
     * @see KeyStrokeHelper#reregisterKeyStrokeListeners
     */
    @Override
    protected void fireWindowClosing() {
        reregisterKeyStrokeListeners();
        super.fireWindowClosing();
    }

    /**
     * Workaround to force re-registration of the key-stroke listeners on the parent dialog, if any, or the root window
     * otherwise.
     * Failure to do this means that the keyboard shortcuts don't work.
     */
    protected void reregisterKeyStrokeListeners() {
        ApplicationInstance active = ApplicationInstance.getActive();
        if (active != null) {
            WindowPane next = null;
            Window root = active.getDefaultWindow();
            for (Component c : root.getContent().getComponents()) {
                if (c instanceof WindowPane) {
                    WindowPane pane = (WindowPane) c;
                    if (pane.isModal()) {
                        if (pane != this && (next == null || pane.getZIndex() > next.getZIndex())) {
                            next = pane;
                        }
                    }
                }
            }
            if (next != null) {
                KeyStrokeHelper.reregisterKeyStrokeListeners(next);
            } else {
                KeyStrokeHelper.reregisterKeyStrokeListeners(root);
            }
        }
    }

    /**
     * Helper to set the focus.
     *
     * @param component the component to focus on
     */
    protected void setFocus(Component component) {
        FocusHelper.setFocus(component);
    }

}
