/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.echo.dialog;

import nextapp.echo2.app.Button;
import nextapp.echo2.app.Extent;
import nextapp.echo2.app.event.ActionEvent;
import org.openvpms.web.echo.error.ErrorHandler;
import org.openvpms.web.echo.event.ActionListener;
import org.openvpms.web.echo.event.VetoListener;
import org.openvpms.web.echo.event.Vetoable;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.echo.util.StyleSheetHelper;


/**
 * Generic popup dialog, providing OK and Cancel buttons.
 *
 * @author Tim Anderson
 */
public abstract class PopupDialog extends PopupWindow {

    /**
     * OK button identifier.
     */
    public static final String OK_ID = "ok";

    /**
     * Close button identifier.
     */
    public static final String CLOSE_ID = "close";

    /**
     * Apply button identifier.
     */
    public static final String APPLY_ID = "apply";

    /**
     * Cancel button identifier.
     */
    public static final String CANCEL_ID = "cancel";

    /**
     * Yes button identifier.
     */
    public static final String YES_ID = "yes";

    /**
     * No button identifier.
     */
    public static final String NO_ID = "no";

    /**
     * Skip button identifier.
     */
    public static final String SKIP_ID = "skip";

    /**
     * Retry button identifier.
     */
    public static final String RETRY_ID = "retry";

    /**
     * Helper to create a button row containing the OK button.
     */
    public static final String[] OK = {OK_ID};

    /**
     * Helper to create a button row containing the CLOSE button.
     */
    public static final String[] CLOSE = {CLOSE_ID};

    /**
     * Helper to create a button row containing the CANCEL button.
     */
    public static final String[] CANCEL = {CANCEL_ID};

    /**
     * Helper to create a button row containing the OK and CANCEL buttons.
     */
    public static final String[] OK_CANCEL = {OK_ID, CANCEL_ID};

    /**
     * Helper to create a button row containing the OK, SKIP and CANCEL buttons.
     */
    public static final String[] OK_SKIP_CANCEL = {OK_ID, SKIP_ID, CANCEL_ID};

    /**
     * Helper to create a button row containing the OK and SKIP buttons.
     */
    public static final String[] OK_SKIP = {OK_ID, SKIP_ID};

    /**
     * Helper to create a button row containing the SKIP and CANCEL buttons.
     */
    public static final String[] SKIP_CANCEL = {SKIP_ID, CANCEL_ID};

    /**
     * Helper to create a button row containing the YES, and NO buttons.
     */
    public static final String[] YES_NO = {YES_ID, NO_ID};

    /**
     * Helper to create a button row containing the YES, NO and CANCEL buttons.
     */
    public static final String[] YES_NO_CANCEL = {YES_ID, NO_ID, CANCEL_ID};

    /**
     * Helper to create a button row containing the APPLY, OK, and CANCEL
     * buttons.
     */
    public static final String[] APPLY_OK_CANCEL = {APPLY_ID, OK_ID, CANCEL_ID};

    /**
     * Helper to craete a button row containing the RETRY and CANCEL buttons.
     */
    public static final String[] RETRY_CANCEL = {RETRY_ID, CANCEL_ID};

    /**
     * The help context. May be {@code null}
     */
    private final HelpContext help;

    /**
     * The dialog action. May be {@code null}
     */
    private String action;

    /**
     * The listener to veto cancel events. May be {@code null}
     */
    private VetoListener<Vetoable> cancelListener;

    /**
     * The listener to veto skip events. May be {@code null}
     */
    private VetoListener<Vetoable> skipListener;

    /**
     * The default close action.
     */
    private String defaultCloseAction;


    /**
     * Constructs a {@link PopupDialog}.
     *
     * @param title   the window title. May be {@code null}
     * @param buttons the buttons to display
     */
    public PopupDialog(String title, String[] buttons) {
        this(title, null, buttons);
    }

    /**
     * Constructs a {@link PopupDialog}.
     *
     * @param title   the window title. May be {@code null}
     * @param buttons the buttons to display
     * @param help    the help context. May be {@code null}
     */
    public PopupDialog(String title, String[] buttons, HelpContext help) {
        this(title, null, buttons, help);
    }

    /**
     * Constructs a {@link PopupDialog}.
     *
     * @param title   the window title
     * @param style   the window style. May be {@code null}
     * @param buttons the buttons to display
     */
    public PopupDialog(String title, String style, String[] buttons) {
        this(title, style, buttons, null);
    }

    /**
     * Constructs a {@link PopupDialog}.
     *
     * @param title   the window title. May be {@code null}
     * @param style   the window style. May be {@code null}
     * @param buttons the buttons to display
     * @param help    the help context. May be {@code null}
     */
    public PopupDialog(String title, String style, String[] buttons, HelpContext help) {
        this(title, style, buttons, help, null);
    }

    /**
     * Constructs a {@link PopupDialog}.
     *
     * @param title   the window title. May be {@code null}
     * @param style   the window style. May be {@code null}
     * @param buttons the buttons to display
     * @param help    the help context. May be {@code null}
     * @param focus   the focus group. May be {@code null}
     */
    public PopupDialog(String title, String style, String[] buttons, HelpContext help, FocusGroup focus) {
        super(title, style, focus);

        this.help = help;
        for (String button : buttons) {
            addButton(button, false);
        }
        if (buttons.length != 0) {
            defaultCloseAction = buttons[buttons.length - 1];
        }
        if (help != null) {
            getButtons().addKeyListener(help.getKeyCode(), new ActionListener() {
                @Override
                public void onAction(ActionEvent event) {
                    onHelp();
                }
            });
        }
    }

    /**
     * Returns the dialog action.
     *
     * @return the dialog action
     */
    public String getAction() {
        return action;
    }

    /**
     * Sets a listener to veto cancel events.
     *
     * @param listener the listener. May be {@code null}
     */
    public void setCancelListener(VetoListener<Vetoable> listener) {
        cancelListener = listener;
    }

    /**
     * Sets a listener to veto skip events.
     *
     * @param listener the listener. May be {@code null}
     */
    public void setSkipListener(VetoListener<Vetoable> listener) {
        skipListener = listener;
    }

    /**
     * Sets the default action when the close button is pressed.
     * <p/>
     * Defaults to the last button displayed.
     *
     * @param action the default action. May be {@code null}
     */
    public void setDefaultCloseAction(String action) {
        defaultCloseAction = action;
    }

    /**
     * Processes a user request to close the window (via the close button).
     * <p/>
     * If there is an {@link #defaultCloseAction}, this will be invoked.
     */
    @Override
    public void userClose() {
        if (action == null && defaultCloseAction != null) {
            onButtonProtected(defaultCloseAction);
        } else {
            onClosing();
            super.userClose();
        }
    }

    /**
     * Returns the help context.
     *
     * @return the help context. May be {@code null}
     */
    public HelpContext getHelpContext() {
        return help;
    }

    /**
     * Sets the action and closes the window.
     *
     * @param action the action
     */
    public void close(String action) {
        setAction(action);
        close();
    }

    /**
     * Resizes the dialog if required.
     * <p/>
     * This looks for a property with the specified name with a value of the format {@code <width>x<height>}. <br/>
     * The width and height are expressed in terms of the font units and will be converted to pixels using the font
     * size.
     *
     * @param property the size property name
     * @return {@code true} if the size was updated
     */
    public boolean resize(String property) {
        boolean resized = false;
        int width = -1;
        int height = -1;
        String size = StyleSheetHelper.getProperty(property);
        if (size != null) {
            String[] values = size.split("x");
            if (values.length == 2) {
                try {
                    width = Integer.parseInt(values[0]);
                    height = Integer.parseInt(values[1]);
                    int fontSize = StyleSheetHelper.getProperty("font.size", -1);
                    width *= fontSize;
                    height *= fontSize;
                } catch (NumberFormatException ignore) {
                    // do nothing
                }
            }
            if (width > 0 && height > 0) {
                setWidth(new Extent(width));
                setHeight(new Extent(height));
                resized = true;
            }
        }
        return resized;
    }

    /**
     * Sets the dialog action.
     *
     * @param action the action
     */
    protected void setAction(String action) {
        this.action = action;
    }

    /**
     * Invoked when a button is pressed. This delegates to the appropriate
     * on*() method for the button if it is known, else sets the action to
     * the button identifier and closes the window.
     *
     * @param button the button identifier
     */
    protected void onButton(String button) {
        try {
            if (OK_ID.equals(button)) {
                onOK();
            } else if (CLOSE_ID.equals(button)) {
                onClose();
            } else if (CANCEL_ID.equals(button)) {
                onCancel();
            } else if (YES_ID.equals(button)) {
                onYes();
            } else if (NO_ID.equals(button)) {
                onNo();
            } else if (SKIP_ID.equals(button)) {
                onSkip();
            } else if (APPLY_ID.equals(button)) {
                onApply();
            } else if (RETRY_ID.equals(button)) {
                onRetry();
            } else {
                setAction(button);
                close();
            }
        } catch (Throwable exception) {
            ErrorHandler.getInstance().error(exception);
        }
    }

    /**
     * Invoked when the 'OK' button is pressed. This sets the action and closes the window.
     */
    protected void onOK() {
        close(OK_ID);
    }

    /**
     * Invoked when the 'close' button is pressed. This sets the action to CLOSE and closes the window.
     */
    protected void onClose() {
        close(CLOSE_ID);
    }

    /**
     * Invoked when the 'cancel' button is pressed. If a {@link VetoListener}
     * has been registered, this will be notified, otherwise {@link #doCancel} will be invoked.
     */
    protected void onCancel() {
        if (cancelListener != null) {
            cancelListener.onVeto(veto -> {
                if (!veto) {
                    doCancel();
                }
            });
        } else {
            doCancel();
        }
    }

    /**
     * Cancels the operation.
     * <p/>
     * This implementation closes the dialog, setting the action to {@link #CANCEL_ID}.
     */
    protected void doCancel() {
        close(CANCEL_ID);
    }

    /**
     * Invoked when the 'yes' button is pressed. This sets the action and closes
     * the window.
     */
    protected void onYes() {
        close(YES_ID);
    }

    /**
     * Invoked when the 'no' button is pressed. This sets the action and closes
     * the window.
     */
    protected void onNo() {
        close(NO_ID);
    }

    /**
     * Invoked when the 'skip' button is pressed. If a {@link VetoListener}
     * has been registered, this will be notified, otherwise {@link #doSkip()} will be invoked.
     */
    protected void onSkip() {
        if (skipListener != null) {
            skipListener.onVeto(veto -> {
                if (!veto) {
                    doSkip();
                }
            });
        } else {
            doSkip();
        }
    }

    /**
     * Invoked when the 'skip' button is pressed. This sets the action and
     * closes the window.
     */
    protected void doSkip() {
        close(SKIP_ID);
    }

    /**
     * Invoked when the 'apply' button is pressed. This sets the action and
     * closes the window.
     */
    protected void onApply() {
        close(APPLY_ID);
    }

    /**
     * Invoked when the 'retry' button is pressed. This sets the action and
     * closes the window.
     */
    protected void onRetry() {
        close(RETRY_ID);
    }

    /**
     * Invoked when the help button is pressed.
     */
    protected void onHelp() {
        HelpContext help = getHelpContext();
        if (help != null) {
            help.show();
        }
    }

    /**
     * Invoked just prior to the dialog closing.
     * <p/>
     * This implementation is a no-op.
     */
    protected void onClosing() {
    }

    /**
     * Adds a new button.
     *
     * @param id the button identifier
     * @return the new button
     */
    protected Button addButton(final String id) {
        return addButton(id, false);
    }

    /**
     * Adds a new button.
     *
     * @param id              the button identifier
     * @param disableShortcut if {@code true} disable any keyboard shortcut
     * @return the new button
     */
    protected Button addButton(String id, boolean disableShortcut) {
        return addButton(id, disableShortcut, () -> onButtonProtected(id));
    }


    /**
     * Resizes the dialog if required.
     * <p/>
     * This implementation looks for a style sheet property name {@code <styleName>.size} with a value of the format
     * {@code <width>x<height>}<br/>
     * The width and height are expressed in terms of the font units that will be converted to pixels using the font
     * size.
     */
    protected void resize() {
        String styleName = getStyleName();
        if (styleName != null) {
            resize(styleName + ".size");
        }
    }

    /**
     * Invokes {@link #onButton(String)}, catching exceptions.
     * <p/>
     * Note that this ignores any event if there are other modal dialogs displayed above the dialog. This is to handle
     * the case where an echo ChangeEvent is processed that generates dialogs, before the button action listener is
     * triggered.
     *
     * @param id the button identifier
     */
    private void onButtonProtected(String id) {
        if (!DialogManager.isHidden(this)) {
            try {
                onButton(id);
            } catch (Throwable exception) {
                ErrorHandler.getInstance().error(exception);
            }
        }
    }

}
