/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2019 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.echo.frame;

import echopointng.HttpPaneEx;
import nextapp.echo2.app.Component;
import nextapp.echo2.app.event.ActionEvent;
import nextapp.echo2.app.event.ActionListener;

import java.util.EventListener;

/**
 * Displays an iframe, and invokes an {@code ActionListener} when the frame receives a message event.
 * <p/>
 * NOTE: message events need to captured at the browser window level, so a {@link MonitoredFrame} may receive events
 * generated by other frames.<br/>
 * The {@link ActionEvent#getActionCommand()} will contain the message.
 * <p/>
 * Only those events generated from the same origin as the specified URI will be passed.
 *
 * @author Tim Anderson
 */
public class MonitoredFrame extends HttpPaneEx {

    /**
     * The message property.
     */
    static final String MESSAGE = "message";

    /**
     * Constructs a {@link MonitoredFrame}.
     *
     * @param uri the uri to display
     */
    public MonitoredFrame(String uri) {
        setURI(uri);
    }

    /**
     * Adds an {@code ActionListener} to the frame.
     * <p/>
     * It will be invoked when the message event is received.
     *
     * @param listener the {@code ActionListener} to add
     */
    public void addActionListener(ActionListener listener) {
        getEventListenerList().addListener(ActionListener.class, listener);
    }

    /**
     * Removes an {@code ActionListener.
     *
     * @param listener the {@code ActionListener to remove
     */
    public void removeActionListener(ActionListener listener) {
        getEventListenerList().removeListener(ActionListener.class, listener);
    }

    /**
     * Processes input.
     *
     * @param inputName  the property
     * @param inputValue the property value
     * @see Component#processInput(String, Object)
     */
    @Override
    public void processInput(String inputName, Object inputValue) {
        if (inputName.equals(MESSAGE)) {
            fireActionEvent((inputValue) != null ? inputValue.toString() : null);
        } else {
            super.processInput(inputName, inputValue);
        }
    }

    /**
     * Fires an action event to all listeners.
     */
    private void fireActionEvent(String message) {
        if (!hasEventListenerList()) {
            return;
        }
        EventListener[] listeners = getEventListenerList().getListeners(ActionListener.class);
        if (listeners.length != 0) {
            ActionEvent e = new ActionEvent(this, message);
            for (EventListener listener : listeners) {
                ((ActionListener) listener).actionPerformed(e);
            }
        }
    }
}
