/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.mail;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.web.component.im.contact.ContactHelper;
import org.openvpms.web.resource.i18n.Messages;
import org.openvpms.web.system.ServiceHelper;

/**
 * Abstract implementation of {@link AddressFormatter}.
 *
 * @author Tim Anderson
 */
public abstract class AbstractAddressFormatter implements AddressFormatter {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The name node default value.
     */
    private final String defaultValue;

    /**
     * Constructs an {@link AbstractAddressFormatter}.
     */
    public AbstractAddressFormatter() {
        service = ServiceHelper.getArchetypeService();
        defaultValue = ContactHelper.getDefaultEmailName(service);
    }

    /**
     * Returns the email address of a contact.
     *
     * @param contact the email contact. May be {@code null}
     * @return the contact's email address, or {@code null} if none is found
     */
    public String getAddress(Contact contact) {
        return (contact != null) ? ContactHelper.getEmail(contact, service) : null;
    }

    /**
     * Returns the name of a contact. If the contact name has been customised from its default, this will be returned,
     * otherwise the name of the associated party will be returned.
     *
     * @param contact the contact
     * @return the name. May be {@code null}
     */
    public String getName(Contact contact) {
        String name = getContactName(contact);
        if (name == null || StringUtils.equals(defaultValue, name)) {
            name = getPartyName(contact);
        }
        return name;
    }

    /**
     * Returns the qualified name.
     * <p/>
     * This includes the contact name and party name, if a contact name is specified. If not, it just
     * returns the party name.
     *
     * @param contact the email contact
     * @return the qualified name. May be {@code null}
     */
    @Override
    public String getQualifiedName(Contact contact) {
        String name = getContactName(contact);
        String partyName = getPartyName(contact);
        if (name == null || StringUtils.equals(defaultValue, name)) {
            name = partyName;
        } else {
            name = (partyName != null) ? Messages.format("mail.qualifiedname", name, partyName) : name;
        }
        return name;
    }

    /**
     * Returns the contact name and address, or just the address, if the contact doesn't have a name.
     * <p/>
     * If {@code strict} is {@code true}, the returned email address is in RFC822 format. i.e:
     * <pre>
     *   "name" &lt;email address&gt;
     * </pre>
     * If {@code false}, the name is unquoted.
     *
     * @param contact the email contact
     * @param strict  if {@code true}, quote the name to ensure the address conforms to RFC822
     * @return the contact name and address. May {@code null}
     */
    @Override
    public String getNameAddress(Contact contact, boolean strict) {
        String name = getName(contact);
        return getNameAddress(contact, name, strict);
    }

    /**
     * Returns the qualified name and address.
     * <p/>
     * This includes the party name and contact name, if a contact name is specified.
     *
     * @param contact the email contact
     * @return the qualified name and address
     */
    @Override
    public String getQualifiedNameAddress(Contact contact) {
        String name = getQualifiedName(contact);
        return getNameAddress(contact, name, false);
    }

    /**
     * Returns the type of a contact.
     *
     * @param contact the contact
     * @return the type of the contact. May be {@code null}
     */
    @Override
    public String getType(Contact contact) {
        Party party = getParty(contact);
        return party != null ? DescriptorHelper.getDisplayName(party, service) : null;
    }

    /**
     * Returns the name of a contact, trimming any leading or trailing whitespace.
     *
     * @param contact the contact
     * @return the contact name. May be {@code null)}
     */
    private String getContactName(Contact contact) {
        return StringUtils.trimToNull(contact.getName());
    }

    /**
     * Returns the name of the party associated with a contact, if any.
     *
     * @param contact the contact
     * @return the party name. May be {@code null}
     */
    private String getPartyName(Contact contact) {
        Party party = getParty(contact);
        return (party != null) ? StringUtils.trimToNull(party.getName()) : null;
    }

    /**
     * Returns the party associated with a contact.
     *
     * @param contact the contact
     * @return the party. May be {@code null}
     */
    private Party getParty(Contact contact) {
        // really don't want to expose getParty() on the interface so need to cast. TODO
        return ((org.openvpms.component.business.domain.im.party.Contact) contact).getParty();
    }

    /**
     * Returns the contact name and address.
     *
     * @param contact the contact
     * @param name    the contact name
     * @param strict  if {@code true}, quote the name to ensure the address is valid RFC822
     * @return the contact name and address
     */
    private String getNameAddress(Contact contact, String name, boolean strict) {
        String address = getAddress(contact);
        return new EmailAddress(address, name).toString(strict);
    }

}
