/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.report;

import org.openvpms.archetype.rules.doc.DocumentTemplate;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.document.Document;
import org.openvpms.report.DocFormats;
import org.openvpms.report.IMReport;
import org.openvpms.report.ParameterType;
import org.openvpms.report.PrintProperties;
import org.openvpms.report.Report;
import org.openvpms.report.ReportException;

import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import static org.openvpms.report.Report.IS_EMAIL;


/**
 * Generates {@link Document}s from one or more objects, using a {@link IMReport}.
 *
 * @author Tim Anderson
 */
public abstract class Reporter<T> {

    /**
     * The default document format.
     */
    public static final String DEFAULT_MIME_TYPE = DocFormats.PDF_TYPE;

    /**
     * The objects to generate the document from.
     */
    private final Iterable<T> objects;

    /**
     * The object to generate the document from, or {@code null} if the
     * document is being generated from a collection.
     */
    private final T object;

    /**
     * The parameters to pass to the report.
     */
    private Map<String, Object> parameters = new HashMap<>();

    /**
     * The fields to pass to the report.
     */
    private Map<String, Object> fields;

    /**
     * Constructs a {@link Reporter} to generate documents from a single object.
     *
     * @param object the object
     */
    public Reporter(T object) {
        objects = Collections.singletonList(object);
        this.object = object;
    }

    /**
     * Constructs a new {@link Reporter} to generate documents from a collection of objects.
     *
     * @param objects the objects
     */
    public Reporter(Iterable<T> objects) {
        this.objects = objects;
        object = null;
    }

    /**
     * Returns the object that the document is being generated from.
     *
     * @return the object, or {@code null} if the document is being generated from a collection
     */
    public T getObject() {
        return object;
    }

    /**
     * Returns the objects.
     *
     * @return the objects
     */
    public Iterable<T> getObjects() {
        return objects;
    }

    /**
     * Creates the document.
     * <p>
     * Documents are formatted according to the default mime type. If the document has an {@link Report#IS_EMAIL}
     * parameter, then this will be set {@code false}.
     *
     * @return the document
     * @throws OpenVPMSException for any error
     */
    public Document getDocument() {
        return getDocument(null, false);
    }

    /**
     * Creates the document, in the specified mime type.
     *
     * @param type  the mime type. If {@code null} the default mime type associated with the report will be used.
     * @param email if {@code true} indicates that the document will be emailed. Documents generated from templates
     *              can perform custom formatting
     * @return the document
     * @throws OpenVPMSException for any error
     */
    public Document getDocument(String type, boolean email) {
        IMReport<T> report = getReport();
        if (type == null) {
            type = report.getDefaultMimeType();
        }
        String mimeType = type;
        Map<String, Object> map = new HashMap<>(getParameters(email));
        Document document = generate(report, () -> report.generate(getObjects(), map, fields, mimeType));
        setName(document);
        return document;
    }

    /**
     * Determines if printing is supported.
     *
     * @return {@code true} if printing is supported, otherwise {@code false}
     */
    public boolean canPrint() {
        return getReport().canPrint();
    }

    /**
     * Prints the report.
     *
     * @param properties the print properties
     */
    public void print(PrintProperties properties) {
        IMReport<T> report = getReport();
        generate(report, () -> report.print(getObjects(), getParameters(false), fields, properties));
    }

    /**
     * Generates a report for a collection of objects to the specified stream.
     *
     * @param type  the mime type. If {@code null} the default mime type associated with the report will be used.
     * @param email if {@code true} indicates that the document will be emailed. Documents generated from templates
     *              can perform custom formatting
     * @throws ReportException               for any report error
     * @throws ArchetypeServiceException     for any archetype service error
     * @throws UnsupportedOperationException if this operation is not supported
     */
    public void generate(String type, boolean email, OutputStream stream) {
        IMReport<T> report = getReport();
        if (type == null) {
            type = report.getDefaultMimeType();
        }
        String mimeType = type;
        Map<String, Object> map = new HashMap<>(getParameters(email));
        generate(report, () -> report.generate(getObjects(), map, fields, mimeType, stream));
    }

    /**
     * Sets parameters to pass to the report.
     *
     * @param parameters a map of parameter names and their values, to pass to
     *                   the report. May be {@code null}
     */
    public void setParameters(Map<String, Object> parameters) {
        this.parameters = parameters;
    }

    /**
     * Returns a map of parameters names and their values, to pass to the
     * report.
     *
     * @return a map of parameter names and their values. May be {@code null}
     */
    public Map<String, Object> getParameters() {
        return parameters;
    }

    /**
     * Returns the set of parameter types that may be supplied to the report.
     * <p>
     * This suppresses return of the {@link Report#IS_EMAIL} parameter as this is dealt with automatically.
     *
     * @return the parameter types
     */
    public Set<ParameterType> getParameterTypes() {
        Set<ParameterType> result = new LinkedHashSet<>();
        for (ParameterType type : getReport().getParameterTypes()) {
            if (!IS_EMAIL.equals(type.getName())) {
                result.add(type);
            }
        }
        return result;
    }

    /**
     * Sets the fields to pass the report.
     *
     * @param fields the fields. May be {@code null}
     */
    public void setFields(Map<String, Object> fields) {
        this.fields = fields;
    }

    /**
     * Returns the document template.
     *
     * @return the document template, or {@code null} if this report isn't generated from a template
     * @throws ArchetypeServiceException for any archetype service error
     */
    public DocumentTemplate getTemplate() {
        return null;
    }

    /**
     * Returns the report.
     *
     * @return the report
     * @throws OpenVPMSException         for any error
     * @throws ReportException           for any report error
     * @throws ArchetypeServiceException for any archetype service error
     */
    protected abstract IMReport<T> getReport();

    /**
     * Returns the report parameters.
     *
     * @param email if the report has an {@link Report#IS_EMAIL} parameter, then this will be supplied with the value of
     *              {@code email}. This enables reports to be customised for email vs printing.
     * @return the report parameters
     */
    protected Map<String, Object> getParameters(boolean email) {
        Map<String, Object> result;
        if (getReport().hasParameter(IS_EMAIL)) {
            result = new HashMap<>();
            if (parameters != null) {
                result.putAll(parameters);
            }
            result.put(IS_EMAIL, email);
        } else {
            result = parameters;
        }
        return result;
    }

    /**
     * Updates the document name.
     * <p>
     * This can be used to update the document name from its default value, prior to returning it to the caller.
     * <p>
     * This implementation is a no-op.
     *
     * @param document the document to update
     */
    protected void setName(Document document) {
        // no-op
    }

    /**
     * Generates a document, performing performance logging if logging is enabled.
     *
     * @param report    the report
     * @param generator the report generator
     * @return the generated document
     */
    private Document generate(IMReport<T> report, Supplier<Document> generator) {
        return new ReportRunner(report, object).run(generator);
    }

    /**
     * Generates a document, performing performance logging if logging is enabled.
     *
     * @param report    the report
     * @param generator the report generator
     */
    private void generate(IMReport<T> report, Runnable generator) {
        new ReportRunner(report, object).run(generator);
    }

}
