/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.query;

import org.apache.commons.collections4.CollectionUtils;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.act.Participation;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.query.criteria.CriteriaBuilder;
import org.openvpms.component.query.criteria.Path;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.BaseArchetypeConstraint;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IConstraint;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.JoinConstraint;
import org.openvpms.component.system.common.query.ParticipationConstraint;
import org.openvpms.component.system.common.query.RelationalOp;
import org.openvpms.component.system.common.query.ShortNameConstraint;
import org.openvpms.web.system.ServiceHelper;

import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;

import static org.openvpms.component.system.common.query.Constraints.and;
import static org.openvpms.component.system.common.query.Constraints.gte;
import static org.openvpms.component.system.common.query.Constraints.isNull;
import static org.openvpms.component.system.common.query.Constraints.lte;
import static org.openvpms.component.system.common.query.Constraints.or;


/**
 * Query helper.
 *
 * @author Tim Anderson
 */
public class QueryHelper {

    /**
     * Determines if a node is a participation node.
     *
     * @param descriptor the node descriptor
     * @return {@code true} if the node is a participation node
     */
    public static boolean isParticipationNode(NodeDescriptor descriptor) {
        return descriptor.isCollection() && "/participations".equals(descriptor.getPath());
    }

    /**
     * Helper to return the descriptor referred to by a constraint.
     *
     * @param archetypes the archetype constraint
     * @param node       the node name
     * @return the corresponding descriptor or {@code null}
     */
    public static NodeDescriptor getDescriptor(ShortNameConstraint archetypes, String node) {
        String[] shortNames = archetypes.getShortNames();
        if (shortNames.length > 0) {
            ArchetypeDescriptor archetype = ServiceHelper.getArchetypeService().getArchetypeDescriptor(shortNames[0]);
            if (archetype != null) {
                return archetype.getNodeDescriptor(node);
            }
        }
        return null;
    }

    /**
     * Adds a sort constraint on a participation node.
     *
     * @param acts       the act short names constraint. Must specify an alias
     * @param query      the query. Must reference {@code acts}
     * @param descriptor the participation node descriptor
     * @param ascending  if {@code true} sort ascending
     */
    public static void addSortOnParticipation(ShortNameConstraint acts,
                                              ArchetypeQuery query,
                                              NodeDescriptor descriptor,
                                              boolean ascending) {
        JoinConstraint particJoin = Constraints.leftJoin(descriptor.getName(), getAlias(descriptor.getName(), query));
        JoinConstraint entityJoin = Constraints.leftJoin("entity", getAlias("entity", query));

        particJoin.add(entityJoin);
        acts.add(particJoin);
        query.add(Constraints.sort(entityJoin.getAlias(), "name", ascending));
    }

    /**
     * Determines if a result set selects an object, using a linear search.
     *
     * @param set       the query
     * @param reference the object reference to check
     * @return {@code true} if the query selects the reference; otherwise {@code false}
     */
    public static <T extends IMObject> boolean selects(ResultSet<T> set, Reference reference) {
        boolean result = false;
        Iterator<T> iter = new ResultSetIterator<>(set);
        while (iter.hasNext()) {
            if (iter.next().getObjectReference().equals(reference)) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Returns all objects matching the specified query in a list.
     *
     * @param query the query
     * @return the matching objects
     */
    public static <T extends IMObject> List<T> query(ArchetypeQuery query) {
        return query(query, ServiceHelper.getArchetypeService());
    }

    /**
     * Returns all objects matching the specified query in a list.
     *
     * @param query   the query
     * @param service the archetype service
     * @return the matching objects
     */
    public static <T extends IMObject> List<T> query(ArchetypeQuery query, IArchetypeService service) {
        List<T> matches = new ArrayList<>();
        CollectionUtils.addAll(matches, new IMObjectQueryIterator<>(service, query));
        return matches;
    }

    /**
     * Returns all objects matching the specified query in a list.
     *
     * @param query the query
     * @return the matching objects
     */
    public static <T extends IMObject> List<T> query(Query<T> query) {
        List<T> matches = new ArrayList<>();
        ResultSet<T> set = query.query();
        if (set != null) {
            CollectionUtils.addAll(matches, new ResultSetIterator<>(set));
        }
        return matches;
    }

    /**
     * Returns all objects matching the specified short names, sorted on the specified nodes.
     *
     * @param shortNames the archetype short names
     * @param sortNodes  the sort nodes
     * @return the matching objects
     */
    public static <T extends IMObject> List<T> query(String[] shortNames, String... sortNodes) {
        ArchetypeQuery query = new ArchetypeQuery(shortNames, false, true);
        for (String sort : sortNodes) {
            query.add(Constraints.sort(sort));
        }
        return query(query);
    }

    /**
     * Helper to create a date range constraint for acts, on a particular date.
     *
     * @param date the date
     * @return a new constraint
     */
    public static IConstraint createDateRangeConstraint(Date date) {
        return createDateRangeConstraint(date, "startTime", "endTime");
    }

    /**
     * Helper to create a date range constraint on two nodes, where the end time may be null.
     *
     * @param date the date
     * @param from the from-node name
     * @param to   the to-node name
     * @return a new constraint
     */
    public static IConstraint createDateRangeConstraint(Date date, String from, String to) {
        return and(lte(from, date), or(gte(to, date), isNull(to)));
    }

    /**
     * Helper to create a date range predicate on two nodes, where the to-time may be null.
     *
     * @param date    the date
     * @param from    the from-node name
     * @param to      the to-node name
     * @param builder the criteria builder
     * @return a new predicate
     */
    public static Predicate createDateRangePredicate(Date date, Path<Date> from, Path<Date> to,
                                                     CriteriaBuilder builder) {
        return builder.and(
                builder.lessThanOrEqualTo(from, date),
                builder.or(
                        builder.greaterThanOrEqualTo(to, date),
                        builder.isNull(to)));
    }

    /**
     * Helper to create a date range constraint for acts of the form:<br/>
     * {@code act.startTime <= from && (act.endTime >= from || act.endTime == null)}
     *
     * @param from the from date
     * @param to   the to date. May be {@code null}
     * @return a new constraint
     */
    public static IConstraint createDateRangeConstraint(Date from, Date to) {
        IConstraint result;
        if (to != null) {
            result = or(createDateRangeConstraint(from), createDateRangeConstraint(to));
        } else {
            result = or(gte("startTime", from), createDateRangeConstraint(from));
        }
        return result;
    }

    /**
     * Helper to create a date range constraint for a participation of the form:<br/>
     * {@code participation.startTime <= from && (participation.endTime >= from || participation.endTime == null)}
     *
     * @param from the from date
     * @param to   the to date. May be {@code null}
     * @return a new constraint
     */
    public static IConstraint createParticipationDateRangeConstraint(Date from, Date to) {
        IConstraint result;
        if (to != null) {
            result = or(createParticipationDateRangeConstraint(from), createParticipationDateRangeConstraint(to));
        } else {
            ParticipationConstraint gte = new ParticipationConstraint(ParticipationConstraint.Field.StartTime,
                                                                      RelationalOp.GTE, from);
            result = or(gte, createParticipationDateRangeConstraint(from));
        }
        return result;
    }

    /**
     * Helper to create a date range constraint for participations, on a particular date.
     *
     * @param date the date
     * @return a new constraint
     */
    public static IConstraint createParticipationDateRangeConstraint(Date date) {
        ParticipationConstraint lte = new ParticipationConstraint(ParticipationConstraint.Field.StartTime,
                                                                  RelationalOp.LTE, date);
        ParticipationConstraint gte = new ParticipationConstraint(ParticipationConstraint.Field.EndTime,
                                                                  RelationalOp.GTE, date);
        ParticipationConstraint isNull = new ParticipationConstraint(ParticipationConstraint.Field.EndTime,
                                                                     RelationalOp.IS_NULL);
        return and(lte, or(gte, isNull));
    }

    /**
     * Helper to create a constraint on a date node.
     * <p>
     * NOTE: any time component is stripped from the date.
     * <p>
     * If:
     * <ul>
     * <li>{@code from} and {@code to} are {@code null} no constraint is created</li>
     * <li>{@code from} is non-null and {@code to} is {@code null}, a constraint {@code startTime >= from}
     * is returned
     * <li>{@code from} is null and {@code to} is {@code null}, a constraint {@code startTime <= to}
     * is returned
     * <li>{@code from} is non-null and {@code to} is {@code non-null}, a constraint
     * {@code startTime >= from && startTime <= to} is returned
     * </ul>
     *
     * @param from the act from date. May be {@code null}
     * @param to   the act to date, inclusive. May be {@code null}
     * @return a new constraint, or {@code null} if both dates are null
     */
    public static IConstraint createDateConstraint(String node, Date from, Date to) {
        IConstraint result;
        if (from == null && to == null) {
            result = null;
        } else if (from != null && to == null) {
            from = DateRules.getDate(from);
            result = gte(node, from);
        } else if (from == null) {
            to = DateRules.getNextDate(to);
            result = Constraints.lt(node, to);
        } else {
            from = DateRules.getDate(from);
            to = DateRules.getNextDate(to);
            result = and(gte(node, from), Constraints.lt(node, to));
        }
        return result;
    }

    /**
     * Determines if a node is an entityLink node.
     *
     * @param descriptor the node descriptor
     * @return {@code true} if the node is a participation node
     */
    public static boolean isEntityLinkNode(NodeDescriptor descriptor) {
        return descriptor.isCollection() && "/entityLinks".equals(descriptor.getPath());
    }

    /**
     * Adds a sort constraint on an entityLink node.
     *
     * @param acts       the act short names constraint. Must specify an alias
     * @param query      the query. Must reference {@code acts}
     * @param descriptor the participation node descriptor
     * @param ascending  if {@code true} sort ascending
     */
    public static void addSortOnEntityLink(ShortNameConstraint acts, ArchetypeQuery query, NodeDescriptor descriptor,
                                           boolean ascending) {
        JoinConstraint linkJoin = Constraints.leftJoin(descriptor.getName(), getAlias(descriptor.getName(), query));
        JoinConstraint targetJoin = Constraints.leftJoin("target", getAlias("target", query));
        linkJoin.add(targetJoin);
        acts.add(linkJoin);
        query.add(Constraints.sort(targetJoin.getAlias(), "name", ascending));
    }

    /**
     * Returns the page that an object would fall on.
     *
     * @param object     the object to locate
     * @param query      the query. Note that this is modified
     * @param pageSize   the page size
     * @param node       the object node to compare
     * @param ascending  if {@code true} sort on {@code node} in ascending order, else use descending order
     * @param comparator the comparator to compare the key and node values
     * @return the page that an object would fall on, if present
     */
    public static <T extends Comparable> int getPage(IMObject object, ArchetypeQuery query, int pageSize,
                                                     final String node, boolean ascending,
                                                     final Comparator<T> comparator) {
        PageLocator locator = new PageLocator(object, query, pageSize);
        locator.addKey(node, ascending, comparator);
        return locator.getPage();
    }

    /**
     * Helper to create and add a participant constraint to the supplied constraints, if {@code entity} is non-null.
     *
     * @param constraints the constraints to add to
     * @param name        rhe participation node name
     * @param shortName   the participation archetype short name
     * @param object      the entity. May be {@code null}
     */
    public static void addParticipantConstraint(List<ParticipantConstraint> constraints, String name,
                                                String shortName, Entity object) {
        if (object != null) {
            constraints.add(new ParticipantConstraint(name, shortName, object));
        }
    }

    /**
     * Returns a case-insensitive pattern for matching expressions containing wildcards.
     *
     * @param expression the wildcard expression
     * @return a new pattern
     */
    public static Pattern getWildcardPattern(String expression) {
        StringBuilder builder = new StringBuilder(expression.length());
        int length = expression.length();

        // escape any regexp characters
        for (int i = 0; i < length; ++i) {
            char c = expression.charAt(i);
            if ("[](){}.+?$^|#\\".indexOf(c) != -1) {
                builder.append("\\");
            }
            builder.append(c);
        }

        // replace wildcards
        String regex = builder.toString();
        regex = regex.replace("*", ".*?");
        return Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
    }

    /**
     * Returns the first participation linked to an entity.
     *
     * @param entity    the entity
     * @param shortName the participation short name to match on
     * @return the first participation, or {@code null} if none is found
     */
    public static Participation getParticipation(Entity entity, String shortName) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, true, true);
        query.add(Constraints.eq("entity", entity));
        query.add(Constraints.sort("id"));
        query.setFirstResult(0);
        query.setMaxResults(1);
        List<IMObject> rows = query(query);
        return (!rows.isEmpty()) ? (Participation) rows.get(0) : null;
    }

    /**
     * Creates a sorted list of identifiers suitable for use in an {@link Constraints#in} constraint.
     *
     * @param references the object references. Must refer to the same base type
     * @return a sorted list of identifiers
     */
    public static Long[] getIds(List<Reference> references) {
        Long[] result = new Long[references.size()];
        for (int i = 0; i < references.size(); ++i) {
            result[i] = references.get(i).getId();
        }
        Arrays.sort(result);
        return result;
    }

    /**
     * Returns a unique alias for an entity constraint.
     *
     * @param prefix the alias prefix
     * @param query  the archetype query
     * @return the alias
     */
    private static String getAlias(String prefix, ArchetypeQuery query) {
        return prefix + getAliasSuffix(prefix, query.getArchetypeConstraint(), 0);
    }

    private static int getAliasSuffix(String prefix, IConstraint constraint, int maxId) {
        if (constraint instanceof BaseArchetypeConstraint) {
            BaseArchetypeConstraint arch = (BaseArchetypeConstraint) constraint;
            String alias = arch.getAlias();
            if (alias != null && alias.startsWith(prefix)) {
                String suffix = alias.substring(prefix.length());
                try {
                    int id = Integer.parseInt(suffix);
                    if (id > maxId) {
                        maxId = id + 1;
                    }
                } catch (NumberFormatException ignore) {
                    // do nothing
                }
                for (IConstraint child : arch.getConstraints()) {
                    maxId = getAliasSuffix(prefix, child, maxId);
                }
            }
        }
        return maxId;
    }

}
