/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.product;

import org.openvpms.archetype.rules.math.Currency;
import org.openvpms.archetype.rules.math.MathRules;
import org.openvpms.archetype.rules.practice.LocationRules;
import org.openvpms.archetype.rules.product.PricingGroup;
import org.openvpms.archetype.rules.product.ProductArchetypes;
import org.openvpms.archetype.rules.product.ProductPriceRules;
import org.openvpms.archetype.rules.product.ServiceRatioService;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.model.product.ProductPrice;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

/**
 * Abstract implementation of {@link PricingContext}.
 *
 * @author Tim Anderson
 */
public abstract class AbstractPricingContext implements PricingContext {

    /**
     * The currency.
     */
    private final Currency currency;

    /**
     * The practice location. May be {@code null}
     */
    private final Party location;

    /**
     * The price rules.
     */
    private final ProductPriceRules rules;

    /**
     * The service ratio service.
     */
    private final ServiceRatioService serviceRatios;

    /**
     * The pricing group.
     */
    private PricingGroup pricingGroup;

    /**
     * Constructs a {@link AbstractPricingContext}.
     *
     * @param currency      the currency
     * @param location      the practice location. May be {@code null}
     * @param priceRules    the price rules
     * @param locationRules the location rules
     * @param serviceRatios the service ratio service
     */
    public AbstractPricingContext(Currency currency, Party location, ProductPriceRules priceRules,
                                  LocationRules locationRules, ServiceRatioService serviceRatios) {
        this.currency = currency;
        this.pricingGroup = getPricingGroup(location, locationRules);
        this.location = location;
        this.rules = priceRules;
        this.serviceRatios = serviceRatios;
    }

    /**
     * Constructs a {@link AbstractPricingContext}.
     *
     * @param currency      the currency
     * @param pricingGroup  the pricing group
     * @param location      the practice location. May be {@code null}
     * @param priceRules    the price rules
     * @param serviceRatios the service ratio service
     */
    public AbstractPricingContext(Currency currency, PricingGroup pricingGroup, Party location,
                                  ProductPriceRules priceRules, ServiceRatioService serviceRatios) {
        this.currency = currency;
        this.pricingGroup = pricingGroup;
        this.location = location;
        this.rules = priceRules;
        this.serviceRatios = serviceRatios;
    }

    /**
     * Returns the tax-inclusive price given a tax-exclusive price and service ratio.
     * <p>
     * This takes into account customer tax exclusions.
     *
     * @param product      the product
     * @param price        the tax-exclusive price
     * @param serviceRatio the service ratio. May be {@code null}
     * @return the tax-inclusive price, rounded according to the practice currency conventions
     */
    @Override
    public BigDecimal getPrice(Product product, ProductPrice price, BigDecimal serviceRatio) {
        BigDecimal result = BigDecimal.ZERO;
        BigDecimal taxExPrice = price.getPrice();
        if (taxExPrice != null) {
            if (serviceRatio != null) {
                taxExPrice = taxExPrice.multiply(serviceRatio);
            }
            result = rules.getTaxIncPrice(taxExPrice, getTaxRate(product), currency);
        }
        return result;
    }

    /**
     * Returns the fixed prices for a product.
     *
     * @param product the product
     * @param date    the date
     * @return the fixed prices
     */
    @Override
    public List<ProductPrice> getFixedPrices(Product product, Date date) {
        return rules.getProductPrices(product, ProductArchetypes.FIXED_PRICE, date, pricingGroup);
    }

    /**
     * Returns the default fixed price for a product.
     *
     * @param product the product
     * @param date    the date
     * @return the fixed price, or {@code null} if none is found
     */
    @Override
    public ProductPrice getFixedPrice(Product product, Date date) {
        return rules.getProductPrice(product, ProductArchetypes.FIXED_PRICE, date, pricingGroup.getGroup());
    }

    /**
     * Returns the unit price for a product.
     *
     * @param product the product
     * @param date    the date
     * @return the unit price, or {@code null} if none is found
     */
    @Override
    public ProductPrice getUnitPrice(Product product, Date date) {
        return rules.getProductPrice(product, ProductArchetypes.UNIT_PRICE, date, pricingGroup.getGroup());
    }

    /**
     * Returns the first product price with the specified short name and price, active as of the date.
     *
     * @param shortName    the price short name
     * @param price        the tax-inclusive price
     * @param serviceRatio the service ratio, or {@link BigDecimal#ONE} if no ratio applies
     * @param product      the product
     * @param date         the date
     * @return the product price, or {@code null} if none is found
     */
    @Override
    public ProductPrice getProductPrice(String shortName, BigDecimal price, BigDecimal serviceRatio, Product product,
                                        Date date) {
        ProductPrice result = null;
        List<ProductPrice> prices = rules.getProductPrices(product, shortName, date, pricingGroup);
        for (ProductPrice p : prices) {
            if (getPrice(product, p, BigDecimal.ONE).compareTo(price) == 0) {
                result = p;
                break;
            }
        }
        return result;
    }

    /**
     * Returns the tax-exclusive price for a product on a date.
     * <p/>
     * This uses:
     * <ul>
     *     <li>the default fixed price; and </li>
     *     <li>the unit price; and</li>
     *     <li>a quantity of {@code 1}</li>
     *     <li>a service ratio of {@code 1}</li>
     * </ul>
     * At least one of the fixed or unit price must be present.
     *
     * @param product the product
     * @param date    the date
     * @return the price, or {@code null} if no fixed or unit price is available
     */
    @Override
    public BigDecimal getTaxExPrice(Product product, Date date) {
        BigDecimal result = null;
        BigDecimal unitPrice = getPrice(getUnitPrice(product, date));
        BigDecimal fixedPrice = getPrice(getFixedPrice(product, date));
        if (unitPrice != null || fixedPrice != null) {
            result = MathRules.calculateTotal(fixedPrice != null ? fixedPrice : BigDecimal.ZERO,
                                              unitPrice != null ? unitPrice : BigDecimal.ZERO,
                                              BigDecimal.ONE, BigDecimal.ZERO, 2);
        }
        return result;
    }

    /**
     * Returns the service ratio for a product, department and date.
     *
     * @param product    the product
     * @param department the department. May be {@code null}
     * @param date       the date
     * @return the service ratio, or {@code null} if none is defined
     */
    public BigDecimal getServiceRatio(Product product, Entity department, Date date) {
        BigDecimal result = null;
        if (product != null && location != null) {
            result = serviceRatios.getServiceRatio(product, department, location, date);
        }
        return result;
    }

    /**
     * Sets the pricing group.
     *
     * @param group the pricing group
     */
    public void setPricingGroup(PricingGroup group) {
        this.pricingGroup = group;
    }

    /**
     * Returns the pricing group.
     *
     * @return the pricing group
     */
    public PricingGroup getPricingGroup() {
        return pricingGroup;
    }

    /**
     * Returns the currency.
     *
     * @return the currency
     */
    protected Currency getCurrency() {
        return currency;
    }

    /**
     * Returns the price rules.
     *
     * @return the price rules
     */
    protected ProductPriceRules getRules() {
        return rules;
    }

    /**
     * Returns the tax rate for a product, minus any tax exclusions.
     *
     * @param product the product
     * @return the product tax rate
     */
    protected abstract BigDecimal getTaxRate(Product product);

    /**
     * Determines the pricing group from the location.
     *
     * @param location the location. May be {@code null}
     * @return the pricing group. May be {@code null}
     */
    protected PricingGroup getPricingGroup(Party location, LocationRules rules) {
        Lookup result = null;
        if (location != null) {
            result = rules.getPricingGroup(location);
        }
        return new PricingGroup(result);
    }

    /**
     * Returns the price associated with a product price.
     *
     * @param price the product price. May be {@code null}
     * @return the price. May be {@code null}
     */
    private BigDecimal getPrice(ProductPrice price) {
        return price != null ? price.getPrice() : null;
    }
}
