/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.view;

import nextapp.echo2.app.Component;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.web.component.bound.BoundTextComponentFactory;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.LookupNameHelper;
import org.openvpms.web.component.im.view.layout.ViewLayoutStrategyFactory;
import org.openvpms.web.component.property.Property;
import org.openvpms.web.echo.factory.ComponentFactory;
import org.openvpms.web.echo.factory.TextComponentFactory;
import org.openvpms.web.echo.style.Styles;
import org.openvpms.web.echo.text.TextComponent;
import org.openvpms.web.echo.text.TextField;
import org.openvpms.web.resource.i18n.format.DateFormatter;
import org.openvpms.web.system.ServiceHelper;

import java.text.DateFormat;


/**
 * An {@link IMObjectComponentFactory} that returns read-only components.
 *
 * @author Tim Anderson
 */
public class ReadOnlyComponentFactory extends AbstractReadOnlyComponentFactory {

    /**
     * The maximum no. of columns to display for lookup fields.
     */
    public static final int MAX_DISPLAY_LENGTH = 50;

    /**
     * Constructs a {@link ReadOnlyComponentFactory}.
     *
     * @param context the layout context
     */
    public ReadOnlyComponentFactory(LayoutContext context) {
        this(context, Styles.DEFAULT);
    }

    /**
     * Constructs a {@link ReadOnlyComponentFactory}.
     *
     * @param context the layout context
     * @param style   the style name to use
     */
    public ReadOnlyComponentFactory(LayoutContext context, String style) {
        super(context, ServiceHelper.getBean(ViewLayoutStrategyFactory.class), style);
    }

    /**
     * Helper to create a read-only text field that has a width based on the number of characters present.
     * <p/>
     * This selects a width that displays the text in field slightly bigger than the text, up to maxLength characters.
     *
     * @param text      the text. May be {@code null}
     * @param minLength the minimum length of the field, if the text is non-null
     * @param maxLength the maximum display length
     * @param style     the component style name
     * @return a new text field
     */
    public static TextComponent getText(String text, int minLength, int maxLength, String style) {
        TextComponent result;
        if (text != null && !text.isEmpty()) {
            minLength = text.length();
        } else if (minLength > 20) {
            minLength = 20; // value is empty, so shrink the display
        }

        result = TextComponentFactory.create(text, minLength, maxLength);
        ComponentFactory.setStyle(result, style);
        result.setEnabled(false);
        return result;
    }

    /**
     * Returns a component to display a lookup.
     *
     * @param property the property
     * @param context  the context object
     * @return a component to display the lookup
     */
    protected Component createLookup(Property property, IMObject context) {
        TextComponent result;
        String value = LookupNameHelper.getName(context, property.getName());
        result = getText(value, property.getMaxLength(), MAX_DISPLAY_LENGTH, getStyle());
        return result;
    }

    /**
     * Returns a component to display a date.
     *
     * @param property the property to bind the field to
     * @return a component to display the date
     */
    protected Component createDate(Property property) {
        DateFormat format = DateFormatter.getDateFormat(false);
        int maxColumns = DateFormatter.getLength(format);
        TextField result = BoundTextComponentFactory.create(property, maxColumns, format);
        ComponentFactory.setStyle(result, getStyle());
        return result;
    }

}
