/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.edit;

import nextapp.echo2.app.Component;
import nextapp.echo2.app.Label;
import nextapp.echo2.app.Row;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.IMObjectCreationListener;
import org.openvpms.web.component.im.util.IMObjectCreator;
import org.openvpms.web.component.property.CollectionProperty;
import org.openvpms.web.component.property.Modifiable;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.echo.factory.LabelFactory;
import org.openvpms.web.echo.focus.FocusGroup;
import org.openvpms.web.resource.i18n.Messages;

import java.util.List;


/**
 * Editor for collections of {@link IMObject}s with 0..1 or 1..1 cardinality.
 * <p/>
 * Note that for collections of 0..1 cardinality, the child editor is created and destroyed on demand, so any
 * listener
 *
 * @author Tim Anderson
 */
public abstract class SingleIMObjectCollectionEditor extends AbstractEditableIMObjectCollectionEditor {

    /**
     * The container.
     */
    private Component container;

    /**
     * Constructs a {@link SingleIMObjectCollectionEditor}.
     *
     * @param editor  the collection property
     * @param object  the parent object
     * @param context the layout context
     */
    public SingleIMObjectCollectionEditor(CollectionProperty editor, IMObject object, LayoutContext context) {
        super(editor, object, context);
    }

    /**
     * Constructs a {@link SingleIMObjectCollectionEditor}.
     *
     * @param editor  the collection property editor
     * @param object  the object being edited
     * @param context the layout context
     */
    protected SingleIMObjectCollectionEditor(CollectionPropertyEditor editor, IMObject object, LayoutContext context) {
        super(editor, object, context);
    }

    /**
     * Creates a new object, subject to collection cardinality constraints.
     * <p/>
     * The object is not automatically added to the collection.
     * <p/>
     * If an {@link IMObjectCreationListener} is registered, it will be
     * notified on successful creation of an object.
     *
     * @return {@code null}
     */
    @Override
    public IMObject create() {
        IMObject result = null;
        CollectionPropertyEditor collection = getCollectionPropertyEditor();
        List<IMObject> objects = collection.getObjects();
        if (objects.isEmpty()) {
            String[] shortNames = collection.getArchetypeRange();
            if (shortNames.length == 1) {
                String shortName = shortNames[0];
                result = IMObjectCreator.create(shortName);
                IMObjectCreationListener listener = getCreationListener();
                if (result != null && listener != null) {
                    listener.created(result);
                }
            }
        }
        return result;
    }

    /**
     * Refreshes the collection display.
     */
    public void refresh() {
    }

    /**
     * Returns the focus group.
     *
     * @return the focus group, or {@code null} if the editor hasn't been rendered
     */
    public FocusGroup getFocusGroup() {
        IMObjectEditor editor = getCurrentEditor();
        return (editor != null) ? editor.getFocusGroup() : null;
    }

    /**
     * Lays out the component.
     *
     * @param context the layout context
     * @return the component
     */
    protected Component doLayout(LayoutContext context) {
        container = new Row();
        container.add(createComponent());
        return container;
    }

    /**
     * Validates the object.
     * <p/>
     * This validates the current object being edited, and if valid, the collection.
     *
     * @param validator the validator
     * @return {@code true} if the object and its descendants are valid otherwise {@code false}
     */
    @Override
    protected boolean doValidation(Validator validator) {
        mapObject();
        return isEmpty() || super.doValidation(validator);
    }

    /**
     * Invoked when the collection or an editor changes. Notifies registered listeners.
     * <p/>
     * This implementation ensures that optional collection items are mapped to the collection when they change,
     * using {@link #mapObject}.
     *
     * @param modifiable the modifiable to pass to the listeners
     */
    @Override
    protected void onModified(Modifiable modifiable) {
        super.onModified(modifiable);
        mapObject();
    }

    /**
     * Saves any current edits.
     *
     * @throws OpenVPMSException if the save fails
     */
    @Override
    protected void doSave() {
        mapObject();
        if (isEmpty()) {
            // save the collection, excluding the current editor
            getCollectionPropertyEditor().save();
            IMObjectEditor editor = getCurrentEditor();
            if (editor != null && !editor.getObject().isNew()) {
                // the object has been deleted, so the editor is no longer valid. Need to create a new one.
                boolean hasFocus = editor.getFocusGroup().hasFocus();
                container.removeAll();
                container.add(createComponent());
                if (hasFocus) {
                    // move the focus to the new editor
                    editor = getCurrentEditor();
                    if (editor != null) {
                        editor.getFocusGroup().setFocus();
                    }
                }
            }
        } else {
            super.doSave();
        }
    }

    /**
     * Determines if the object being edited is empty.
     *
     * @return {@code true} if the object is empty
     */
    protected abstract boolean isEmpty();

    /**
     * Adds/removes the object to/from the collection.
     * If the object is empty and the min cardinality is zero, removes the object from the collection otherwise adds it.
     *
     * @return {@code true} if the object is not empty, and has changed
     */
    private boolean mapObject() {
        boolean result = false;
        IMObjectEditor editor = getCurrentEditor();
        if (editor != null) {
            CollectionPropertyEditor collection = getCollectionPropertyEditor();
            if (isEmpty()) {
                collection.remove(editor.getObject());
            } else if (editor.isModified()) {
                result = true;
                collection.add(editor.getObject());
            }
        }
        return result;
    }

    /**
     * Creates the component.
     *
     * @return the component
     */
    private Component createComponent() {
        Component component;
        CollectionPropertyEditor collection = getCollectionPropertyEditor();
        String[] shortNames = collection.getArchetypeRange();
        List<IMObject> objects = collection.getObjects();
        IMObject object;
        String shortName = shortNames[0];
        if (objects.isEmpty()) {
            object = IMObjectCreator.create(shortName);
            IMObjectCreationListener listener = getCreationListener();
            if (object != null && listener != null) {
                listener.created(object);
            }
        } else {
            object = objects.get(0);
        }
        if (object != null) {
            IMObjectEditor editor = getEditor(object);
            setCurrentEditor(editor);
            component = editor.getComponent();
            if (!mapObject() && object.isNew()) {
                // there is a single object present, that is empty. If the object is new, make sure the collection
                // isn't marked as being modified, as the object will be automatically removed prior to save.
                clearModified();
            }
        } else {
            String displayName = DescriptorHelper.getDisplayName(shortName, getService());
            if (displayName == null) {
                displayName = shortName;
            }
            String message = Messages.format("imobject.create.failed", displayName);
            Label label = LabelFactory.create();
            label.setText(message);
            component = label;
        }
        return component;
    }
}
