/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.workflow;

import nextapp.echo2.app.event.WindowPaneEvent;
import org.openvpms.component.exception.OpenVPMSException;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.web.component.im.delete.AbstractIMObjectDeleter;
import org.openvpms.web.component.im.delete.AsyncIMObjectDeletionListener;
import org.openvpms.web.component.im.delete.IMObjectDeletionHandlerFactory;
import org.openvpms.web.component.im.delete.IMObjectDeletionListener;
import org.openvpms.web.component.im.delete.SilentIMObjectDeleter;
import org.openvpms.web.component.im.edit.EditDialog;
import org.openvpms.web.component.im.edit.EditDialogFactory;
import org.openvpms.web.component.im.edit.IMObjectEditor;
import org.openvpms.web.component.im.edit.IMObjectEditorSaver;
import org.openvpms.web.component.im.layout.DefaultLayoutContext;
import org.openvpms.web.component.im.layout.LayoutContext;
import org.openvpms.web.component.im.util.IMObjectHelper;
import org.openvpms.web.component.property.DefaultValidator;
import org.openvpms.web.component.property.Validator;
import org.openvpms.web.component.property.ValidatorError;
import org.openvpms.web.component.util.ErrorHelper;
import org.openvpms.web.echo.dialog.ErrorDialog;
import org.openvpms.web.echo.dialog.ErrorDialogBuilder;
import org.openvpms.web.echo.dialog.PopupDialog;
import org.openvpms.web.echo.event.WindowPaneListener;
import org.openvpms.web.echo.help.HelpContext;
import org.openvpms.web.system.ServiceHelper;


/**
 * Task to edit an {@link IMObject} using an {@link IMObjectEditor}.
 *
 * @author Tim Anderson
 */
public class EditIMObjectTask extends AbstractTask {

    /**
     * Determines if the object should be edited displaying a UI.
     */
    private final boolean interactive;

    /**
     * The object to edit.
     */
    private IMObject object;

    /**
     * The short name of the object to edit.
     */
    private String shortName;

    /**
     * Determines if the object should be created.
     */
    private boolean create;

    /**
     * Determines if editing may be skipped.
     */
    private boolean skip;

    /**
     * Determines if the object should be deleted on cancel or skip.
     */
    private boolean deleteOnCancelOrSkip;

    /**
     * Properties to create the object with.
     */
    private TaskProperties createProperties;

    /**
     * Determines if the UI should be displayed if a the object is invalid.
     * This only applies when {@link #interactive} is {@code false}.
     */
    private boolean showEditorOnError = true;

    /**
     * The current editor.
     */
    private IMObjectEditor editor;

    /**
     * The current edit dialog.
     */
    private EditDialog dialog;


    /**
     * Constructs a new {@code EditIMObjectTask} to edit an object
     * in the {@link TaskContext}.
     *
     * @param shortName the short name of the object to edit
     */
    public EditIMObjectTask(String shortName) {
        this(shortName, false);
    }

    /**
     * Constructs a new {@code EditIMObjectTask}, to edit an object
     * in the {@link TaskContext} or create and edit a new one.
     *
     * @param shortName the object short name
     * @param create    if {@code true}, create the object
     */
    public EditIMObjectTask(String shortName, boolean create) {
        this(shortName, create, true);
    }

    /**
     * Constructs a new {@code EditIMObjectTask}, to edit an object
     * in the {@link TaskContext} or create and edit a new one.
     *
     * @param shortName   the object short name
     * @param create      if {@code true}, create the object
     * @param interactive if {@code true} create an editor and display it;
     *                    otherwise create it but don't display it
     */
    public EditIMObjectTask(String shortName, boolean create,
                            boolean interactive) {
        this.shortName = shortName;
        this.create = create;
        this.interactive = interactive;
    }

    /**
     * Constructs an {@link EditIMObjectTask} to create and edit a new {@code IMObject}.
     *
     * @param shortName        the object short name
     * @param createProperties the properties to create the object with. May be {@code null}
     * @param interactive      if {@code true} create an editor and display it; otherwise create it but don't display it
     */
    public EditIMObjectTask(String shortName, TaskProperties createProperties, boolean interactive) {
        this.shortName = shortName;
        create = true;
        this.createProperties = createProperties;
        this.interactive = interactive;
    }

    /**
     * Constructs an {@link EditIMObjectTask}.
     *
     * @param object the object to edit
     */
    public EditIMObjectTask(IMObject object) {
        this(object, true);
    }

    /**
     * Constructs an {@link EditIMObjectTask}.
     *
     * @param object      the object to edit
     * @param interactive if {@code true} create an editor and display it; otherwise create it but don't display it
     */
    public EditIMObjectTask(IMObject object, boolean interactive) {
        this.object = object;
        this.interactive = interactive;
    }

    /**
     * Determines if editing may be skipped.
     * Note that the object may have changed prior to editing being skipped.
     * Defaults to {@code false}.
     *
     * @param skip if {@code true} editing may be skipped.
     */
    public void setSkip(boolean skip) {
        this.skip = skip;
    }

    /**
     * Determines if the editor should be displayed if the object is invalid.
     * This only applies when non-interactive editing was specified at
     * construction. Defaults to {@code true}.
     *
     * @param show if {@code true} display the editor if the object is invalid
     */
    public void setShowEditorOnError(boolean show) {
        showEditorOnError = show;
    }

    /**
     * Determines if the object should be deleted if the task is cancelled or skipped.
     * <p>
     * Defaults to {@code false}.
     * <p>
     * Note that no checking is performed to see if the object participates in entity relationships before being
     * deleted. To do this, use {@link AbstractIMObjectDeleter} instead.
     *
     * @param delete if {@code true} delete the object on cancel or skip
     */
    public void setDeleteOnCancelOrSkip(boolean delete) {
        deleteOnCancelOrSkip = delete;
    }

    /**
     * Starts the task.
     * <p>
     * The registered {@link TaskListener} will be notified on completion or failure.
     *
     * @param context the task context
     */
    public void start(final TaskContext context) {
        if (object == null) {
            if (create) {
                create(context);
            } else {
                edit(context);
            }
        } else {
            edit(object, context);
        }
    }

    /**
     * Returns the editor.
     *
     * @return the editor, or {@code null} if none exists
     */
    public IMObjectEditor getEditor() {
        return editor;
    }

    /**
     * Returns the edit dialog.
     *
     * @return the edit dialog, or {@code null} if none is being displayed
     */
    public EditDialog getEditDialog() {
        return dialog;
    }

    /**
     * Edits an object located in the context.
     *
     * @param context the task context
     */
    protected void edit(TaskContext context) {
        IMObject object = context.getObject(shortName);
        if (object != null) {
            edit(object, context);
        } else {
            notifyCancelled();
        }
    }

    /**
     * Creates the object.
     * <p/>
     * On completion, delegates to {@link #edit(TaskContext)}.
     *
     * @param context the task context
     */
    protected void create(TaskContext context) {
        CreateIMObjectTask creator = new CreateIMObjectTask(shortName, createProperties);
        creator.addTaskListener(new DefaultTaskListener() {
            public void taskEvent(TaskEvent event) {
                switch (event.getType()) {
                    case SKIPPED:
                        notifySkipped();
                        break;
                    case CANCELLED:
                        notifyCancelled();
                        break;
                    case COMPLETED:
                        edit(context);
                }
            }
        });
        start(creator, context);
    }

    /**
     * Edits an object.
     * <p>
     * Creates a new editor via {@link #createEditor} before delegating
     * to {@link #interactiveEdit}, if editing is interactive, or {@link #backgroundEdit} if not.
     *
     * @param object  the object to edit
     * @param context the task context
     */
    protected void edit(IMObject object, TaskContext context) {
        HelpContext help = context.getHelpContext().topic(object, "edit");
        context = new DefaultTaskContext(context, null, help);

        try {
            editor = createEditor(object, context);
            if (interactive) {
                interactiveEdit(editor, context);
            } else {
                backgroundEdit(editor, context);
            }
        } catch (Throwable exception) {
            onError(object, context, exception);
        }
    }

    /**
     * Invoked when editing fails due to error.
     * <p/>
     * This implementation deletes the object if {@link #deleteOnCancelOrSkip} is {@code true} and invokes
     * {@link #notifyCancelledOnError(Throwable)}.
     *
     * @param object    the object being edited.
     * @param context   the  task context
     * @param exception the cause of the error
     */
    protected void onError(IMObject object, TaskContext context, Throwable exception) {
        if (deleteOnCancelOrSkip) {
            delete(object, context, () -> notifyCancelledOnError(exception));
        } else {
            notifyCancelledOnError(exception);
        }
    }

    /**
     * Creates a new editor for an object.
     *
     * @param object  the object to edit
     * @param context the task context
     * @return a new editor
     */
    protected IMObjectEditor createEditor(IMObject object, TaskContext context) {
        LayoutContext layout = createLayoutContext(context);
        return layout.getEditorFactory().create(object, layout);
    }

    /**
     * Creates an edit layout context.
     *
     * @param context the task context
     * @return a new edit layout context
     */
    protected LayoutContext createLayoutContext(TaskContext context) {
        return new DefaultLayoutContext(true, context, context.getHelpContext());
    }

    /**
     * Shows the editor in an edit dialog.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void interactiveEdit(final IMObjectEditor editor, final TaskContext context) {
        context.setCurrent(object);
        dialog = createEditDialog(editor, skip, context);
        dialog.addWindowPaneListener(new WindowPaneListener() {
            public void onClose(WindowPaneEvent event) {
                context.setCurrent(null);
                String action = dialog.getAction();
                clear();
                onDialogClose(action, editor, context);
            }

        });
        dialog.show();
    }

    /**
     * Invoked when the edit dialog closes to complete the task.
     *
     * @param action  the dialog action
     * @param editor  the editor
     * @param context the task context
     */
    protected void onDialogClose(String action, IMObjectEditor editor, TaskContext context) {
        if (PopupDialog.OK_ID.equals(action)) {
            onEditCompleted();
        } else if (PopupDialog.SKIP_ID.equals(action)) {
            onEditSkipped(editor, context);
        } else {
            onEditCancelled(editor, context);
        }
    }

    /**
     * Attempts to edit an object in the background.<br/>
     * Editing is delegated to {@link #edit(IMObjectEditor, TaskContext)}.
     * <p>
     * If the editor is valid after editing, the object will be saved.
     * If not, and {@link #showEditorOnError} is {@code true}, an interactive edit will occur, otherwise the edit will
     * be cancelled.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void backgroundEdit(IMObjectEditor editor, TaskContext context) {
        context.setCurrent(null);
        editor.getComponent();
        edit(editor, context);
        clear();
        Validator validator = new DefaultValidator();
        if (editor.validate(validator)) {
            backgroundSave(editor, context);
        } else {
            showValidationError(validator);
            if (showEditorOnError) {
                // editor invalid. Pop up an edit dialog.
                interactiveEdit(editor, context);
            }
        }
    }

    /**
     * Saves an edit being performed in the background.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void backgroundSave(IMObjectEditor editor, TaskContext context) {
        IMObjectEditorSaver saver = new IMObjectEditorSaver();
        if (saver.save(editor, () -> backgroundEditFailed(editor, context))) {
            notifyCompleted();
        }
    }

    /**
     * Invoked when saving of the editor fails during a background edit.
     * <p/>
     * This implementation delegates to {@link #onEditCancelled(IMObjectEditor, TaskContext)}
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void backgroundEditFailed(IMObjectEditor editor, TaskContext context) {
        onEditCancelled(editor, context);
    }

    /**
     * Edits an object in the background.
     * This implementation is a no-op.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void edit(IMObjectEditor editor, TaskContext context) {
    }

    /**
     * Creates a new edit dialog.
     *
     * @param editor  the editor
     * @param skip    if {@code true}, editing may be skipped
     * @param context the help context
     * @return a new edit dialog
     */
    protected EditDialog createEditDialog(IMObjectEditor editor, boolean skip, TaskContext context) {
        EditDialog dialog = ServiceHelper.getBean(EditDialogFactory.class).create(editor, context);
        dialog.addSkip(skip);
        return dialog;
    }

    /**
     * Invoked when editing is complete.
     */
    protected void onEditCompleted() {
        clear();
        notifyCompleted();
    }

    /**
     * Invoked when editing is skipped.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void onEditSkipped(IMObjectEditor editor, TaskContext context) {
        clear();
        if (deleteOnCancelOrSkip) {
            delete(editor.getObject(), context, this::notifySkipped);
        } else {
            notifySkipped();
        }
    }

    /**
     * Invoked when editing is cancelled.
     *
     * @param editor  the editor
     * @param context the task context
     */
    protected void onEditCancelled(IMObjectEditor editor, TaskContext context) {
        clear();
        if (deleteOnCancelOrSkip) {
            delete(editor.getObject(), context, this::notifyCancelled);
        } else {
            notifyCancelled();
        }
    }

    /**
     * Deletes an object.
     *
     * @param object   the object to delete
     * @param context  the task context
     * @param callback the callback to invoke on completion or failure
     */
    protected void delete(IMObject object, TaskContext context, Runnable callback) {
        delete(object, context, callback, callback);
    }

    /**
     * Deletes an object.
     *
     * @param object          the object to delete
     * @param context         the task context
     * @param successCallback the callback to invoke if deletion succeeds
     * @param failedCallback  the callback to invoke if deletion fails
     */
    protected void delete(IMObject object, TaskContext context, Runnable successCallback, Runnable failedCallback) {
        if (object.isNew()) {
            successCallback.run();
        } else {
            try {
                object = IMObjectHelper.reload(object);
                if (object != null) {
                    // make sure the last saved instance is being deleted to avoid validation errors
                    IMObjectDeletionHandlerFactory factory
                            = ServiceHelper.getBean(IMObjectDeletionHandlerFactory.class);
                    SilentIMObjectDeleter<IMObject> deleter = new SilentIMObjectDeleter<>(
                            factory, ServiceHelper.getArchetypeService());
                    IMObjectDeletionListener<IMObject> listener = new AsyncIMObjectDeletionListener<IMObject>() {
                        @Override
                        protected void completed() {
                            successCallback.run();
                        }

                        @Override
                        protected void failed() {
                            failedCallback.run();
                        }
                    };
                    deleter.delete(object, context, context.getHelpContext(), listener);
                } else {
                    // object already deleted
                    successCallback.run();
                }
            } catch (OpenVPMSException exception) {
                ErrorHelper.show(exception, failedCallback);
            }
        }
    }

    /**
     * Displays a validation error. If editing is suppressed, this cancels the task on when the dialog closes.
     *
     * @param validator the validator
     */
    private void showValidationError(Validator validator) {
        ValidatorError error = validator.getFirstError();
        String message = (error != null) ? error.toString() : null;
        if (error != null) {
            ErrorDialogBuilder builder = ErrorDialog.newDialog().message(message);
            if (!showEditorOnError) {
                builder.listener(this::notifyCancelled);
            }
            builder.show();
        } else if (!showEditorOnError) {
            notifyCancelled();
        }
    }

    /**
     * Clears state on edit completion/cancellation.
     */
    private void clear() {
        editor = null;
        dialog = null;
    }

}