/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.web.component.im.query;

import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.system.common.query.ArchetypeQueryException;
import org.openvpms.component.system.common.query.BaseArchetypeConstraint;


/**
 * Query implementation that queries {@link IMObject} instances on short name,
 * instance name, and active/inactive status.
 *
 * @author Tim Anderson
 */
public class DefaultQuery<T extends IMObject> extends AbstractIMObjectQuery<T> {

    /**
     * Constructs a {@link DefaultQuery} that queries IMObjects with the specified short names.
     *
     * @param shortNames the short names
     * @param type       the type that this query returns
     * @throws ArchetypeQueryException if the short names don't match any archetypes
     */
    public DefaultQuery(String[] shortNames, Class<T> type) {
        super(shortNames, type);
        setDefaultSortConstraint(NAME_SORT_CONSTRAINT);
    }

    /**
     * Returns the query state.
     *
     * @return the query state
     */
    @Override
    public QueryState getQueryState() {
        return new Memento(this);
    }

    /**
     * Sets the query state.
     *
     * @param state the query state
     */
    @Override
    public void setQueryState(QueryState state) {
        if (state instanceof Memento) {
            Memento memento = (Memento) state;
            setShortName(memento.archetype);
            setValue(memento.value);
            setValueMinLength(memento.valueMinLength);
            setActive(memento.active);
        }
    }

    private static class Memento extends AbstractQueryState {

        private final String archetype;

        private final String value;

        private final int valueMinLength;

        private final BaseArchetypeConstraint.State active;


        /**
         * Constructs a {@link Memento}.
         *
         * @param query the query
         */
        public Memento(DefaultQuery<?> query) {
            super(query);
            archetype = query.getShortName();
            value = query.getValue();
            valueMinLength = query.getValueMinLength();
            active = query.getActive();
        }
    }
}
